<?php
namespace App\Controllers\API;

use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;

class Prescription extends ResourceController
{
    protected $modelName = 'App\Models\Customers_model';
    protected $format = 'json';
    use ResponseTrait;
    
    public function uploadss()
    {
        helper(['form', 'url']);

        // Validate required keys
        $userid = $this->request->getPost('userid');
        $token = $this->request->getHeaderLine('Authorization') ?: $this->request->getPost('token');
        $ip = $this->request->getIPAddress();

        if (!$userid || !$token) {
            return $this->failValidationError('Missing required fields: user_id or token');
        }
        
        
        // Validate file
        $file = $this->request->getFile('prescription');

        if (!$file || !$file->isValid()) {
            return $this->failValidationError('Prescription file is required and must be valid');
        }

        // File type and size validation
        if (!in_array($file->getExtension(), ['jpg', 'jpeg', 'png', 'pdf'])) {
            return $this->failValidationError('Only JPG, PNG, or PDF files are allowed');
        }

        if ($file->getSize() > 10 * 1024 * 1024) {
            return $this->failValidationError('File size must not exceed 10MB');
        }

        // Save the file
        $uploadPath = FCPATH . 'uploads/'.$userid.'/prescriptions';
        $newName = $file->getRandomName();
        $file->move($uploadPath, $newName);
        
        // db
        $db = \Config\Database::connect();
        // Insert into DB
        $db->table('tbl_prescriptions')->insert([
            'customer_id'     => $userid,
            'filename'    => $newName,
            'file_path'   => 'uploads/'.$userid.'/prescriptions',
            'ip_address'  => $ip,
            'token'       => $token,
            'uploaded_at' => date('Y-m-d H:i:s')
        ]);
        return $this->respond([
           'status' => 'success',
           'message' => 'Prescription uploaded successfully',
           'data' => ['userid' => $userid, 'filename' => $uploadPath.'/'.$newName]
        ], 200);
    

        //return $this->respond([
        //'status' => true,
        //'message' => 'Prescription uploaded successfully',
        //'data' => [
        //    'user_id' => $userId,
        //    'ip' => $ip,
        //    'filename' => $newName,
        //    'token' => $token
        //    ]
        //]);
    }
    
    public function upload()
    {
        helper(['form', 'url']);

        // Validate required keys
        $userid = intval($this->request->getPost('userid'));
        $title  = esc($this->request->getPost('title'), 'html');
        $token = esc($this->request->getHeaderLine('Authorization') ?: $this->request->getPost('token'), 'raw');
        $ip = $this->request->getIPAddress();

        if (!$userid || !$token) {
            return $this->failValidationError('Missing required fields: user_id or token');
        }
        
        
        // Validate file
        $file = $this->request->getFile('prescription');

        if (!$file || !$file->isValid()) {
            return $this->failValidationError('Prescription file is required and must be valid');
        }

        // File type and size validation
        if (!in_array($file->getExtension(), ['jpg', 'jpeg', 'png', 'pdf'])) {
            return $this->failValidationError('Only JPG, PNG, or PDF files are allowed');
        }

        if ($file->getSize() > 10 * 1024 * 1024) {
            return $this->failValidationError('File size must not exceed 10MB');
        }

        // Save the file
        $uploadPath = FCPATH . 'uploads/'.$userid.'/prescriptions';
        $newName = $file->getRandomName();
        $file->move($uploadPath, $newName);
        
        // db
        $db = \Config\Database::connect();
        //check if valid customer
        $customer = $db->table('tbl_customers')
            ->select('id')
            ->where('id', $userid)
            ->where('status', 1)
            ->where('deleted', '0')
            ->get()
            ->getRow();
        
        if (!$customer) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid or inactive customer'
            ], 404);
        }
        // Insert into DB
        $db->table('tbl_prescriptions')->insert([
            'customer_id'     => $userid,
            'title'     => $title,
            'filename'    => $newName,
            'file_path'   => 'uploads/'.$userid.'/prescriptions',
            'ip_address'  => $ip,
            'token'       => $token,
            'uploaded_at' => date('Y-m-d H:i:s')
        ]);
        return $this->respond([
           'status' => 'success',
           'message' => 'Prescription uploaded successfully',
           'data' => ['userid' => $userid, 'filepath' => base_url($userid.'/prescriptions/'.$newName)]
        ], 200);
    }
    
    public function listPrescription($customerId = null)
    {
        if (!$customerId || !is_numeric($customerId)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid or missing customer ID.'
            ], 400);
        }
        
        $db = \Config\Database::connect();
        
        $prescriptions = $db->table('tbl_prescriptions')
            ->where('customer_id', $customerId)
            ->where('status', 'pending')
            ->orderBy('uploaded_at', 'DESC')
            ->get()
            ->getResult();
        
        if (empty($prescriptions)) {
            return $this->respond([
                'status' => 'success',
                'message' => 'No prescriptions found.',
                'data' => []
            ]);
        }
        
        $result = array_map(function ($row) {
            return [
                'id'         => $row->id,
                'filename'   => $row->filename,
                'file_url'   => base_url($row->file_path).'/'.$row->filename,
                'uploaded_at'=> $row->uploaded_at
            ];
        }, $prescriptions);
        
        return $this->respond([
            'status' => 'success',
            'message' => 'Prescriptions retrieved successfully',
            'data' => $result
        ]);
    }
    
    public function failValidationErrorsBypass($message)
    {
        return $this->respond([
               'status' => 'error',
               'message' => $message
           ], 400);
    }

    public function uploads()
    {
        $json_data = $this->request->getBody();
        $data = json_decode($json_data, true);

        if (!$this->isValidRequest($json_data)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid Request'
            ], 400);
        }

        $requiredKeys = ['userid', 'token', 'ip'];

        if (!$this->hasRequiredKeys($data, $requiredKeys)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid Request, some keys are missing'
            ], 400);
        }

        $otp = $data["otp"] ?? null;
        $name = $data["name"];
        $phonenumber = $data["phone"];
        $userid = $data["userid"] ?? null;

        if (!$this->isValidPhoneNumber($phonenumber)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid Phone Number'
            ], 400);
        }

        if ($otp == '') {
            return $this->sendOtp($name, $phonenumber);
        } else {
            return $this->verifyOtp($userid, $otp);
        }
    }

    private function isValidRequest($data)
    {
        $contentType = $this->request->getHeaderLine('Content-Type');
        return $contentType == 'application/json' && json_decode($data) !== null;
    }

    private function hasRequiredKeys($data, $requiredKeys)
    {
        foreach ($requiredKeys as $key) {
            if (!array_key_exists($key, $data)) {
                return false;
            }
        }
        return true;
    }

    
    public function deletePrescription()
    {
        $json = $this->request->getBody();
        $data = json_decode($json, true);
    
        if (!is_array($data)) {
            return $this->failValidationError('Invalid JSON input');
        }
    
        $prescriptionId = $data['prescription_id'] ?? null;
        $userId         = $data['userid'] ?? null;
        $token          = $this->request->getHeaderLine('Authorization') ?: ($data['token'] ?? null);
    
        if (!$prescriptionId || !$userId || !$token) {
            return $this->failValidationError('Missing required fields: prescription_id, userid, or token');
        }
    
        $db = \Config\Database::connect();
    
        // Sanitize inputs
        $prescriptionId = intval($prescriptionId);
        $userId = intval($userId);
    
        // Validate customer
        $customer = $db->table('tbl_customers')
            ->select('id')
            ->where('id', $userId)
            ->where('status', 1)
            ->where('deleted', '0')
            ->get()
            ->getRow();
    
        if (!$customer) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid or inactive customer'
            ], 404);
        }
    
        // Fetch prescription
        $prescription = $db->table('tbl_prescriptions')
            ->where('id', $prescriptionId)
            ->where('customer_id', $userId)
            ->get()
            ->getRow();
    
        if (!$prescription) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Prescription not found'
            ], 404);
        }
    
        // Move to history table
        $historyData = (array) $prescription;
        $historyData['deleted_at'] = date('Y-m-d H:i:s');
        $db->table('tbl_prescription_history')->insert($historyData);
    
        // Delete the file
        $filePath = FCPATH . $prescription->file_path . '/' . $prescription->filename;
        if (file_exists($filePath)) {
            @unlink($filePath);
        }
    
        // Delete record
        $db->table('tbl_prescriptions')->where('id', $prescriptionId)->delete();
    
        return $this->respond([
            'status' => 'success',
            'message' => 'Prescription deleted successfully',
            'data' => [
                'prescription_id' => $prescriptionId
            ]
        ], 200);
    }
}
