<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Comment;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\ConfigurationException\RequiredFixerConfigurationException;
use PhpCsFixer\Fixer\Comment\HeaderCommentFixer;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;
use PhpCsFixer\WhitespacesFixerConfig;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Comment\HeaderCommentFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Comment\HeaderCommentFixer>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Comment\HeaderCommentFixer
 */
final class HeaderCommentFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     * @param non-empty-string                 $indent
     * @param non-empty-string                 $lineEnding
     *
     * @dataProvider provideFixCases
     */
    public function testFix(
        array $configuration,
        string $expected,
        ?string $input = null,
        string $indent = '    ',
        string $lineEnding = "\n"
    ): void {
        $this->fixer->setWhitespacesConfig(new WhitespacesFixerConfig($indent, $lineEnding));
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: _AutogeneratedInputConfiguration, 1: string, 2?: string, 3?: non-empty-string, 4?: non-empty-string}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            ['header' => ''],
            '<?php

$a;',
            '<?php

/**
 * new
 */
$a;',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_declare_strict',
            ],
            '<?php
declare(strict_types=1);

/*
 * tmp
 */

namespace A\B;

echo 1;',
            '<?php
declare(strict_types=1);namespace A\B;

echo 1;',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_declare_strict',
                'separate' => 'bottom',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php
declare(strict_types=1);
/**
 * tmp
 */

namespace A\B;

echo 1;',
            '<?php
declare(strict_types=1);

namespace A\B;

echo 1;',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_open',
            ],
            '<?php

/*
 * tmp
 */

declare(strict_types=1);

namespace A\B;

echo 1;',
            '<?php
declare(strict_types=1);

namespace A\B;

echo 1;',
        ];

        yield [
            [
                'header' => 'new',
                'comment_type' => HeaderCommentFixer::HEADER_COMMENT,
            ],
            '<?php

/*
 * new
 */
                ',
            '<?php
                    /** test */
                ',
        ];

        yield [
            [
                'header' => 'new',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * new
 */
                ',
            '<?php
                    /* test */
                ',
        ];

        yield [
            [
                'header' => 'def',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * def
 */
',
            '<?php
',
        ];

        yield [
            ['header' => 'xyz'],
            '<?php

/*
 * xyz
 */

    $b;',
            '<?php
    $b;',
        ];

        yield [
            [
                'header' => 'xyz123',
                'separate' => 'none',
            ],
            '<?php
/*
 * xyz123
 */
    $a;',
            '<?php
    $a;',
        ];

        yield [
            [
                'header' => 'abc',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * abc
 */

$c;',
            '<?php
$c;',
        ];

        yield [
            [
                'header' => 'ghi',
                'separate' => 'both',
            ],
            '<?php

/*
 * ghi
 */

$d;',
            '<?php
$d;',
        ];

        yield [
            [
                'header' => 'ghi',
                'separate' => 'top',
            ],
            '<?php

/*
 * ghi
 */
$d;',
            '<?php
$d;',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_declare_strict',
            ],
            '<?php

/*
 * tmp
 */

declare(ticks=1);

echo 1;',
            '<?php
declare(ticks=1);

echo 1;',
        ];

        yield [
            ['header' => 'Foo'],
            '<?php

/*
 * Foo
 */

echo \'bar\';',
            '<?php echo \'bar\';',
        ];

        yield [
            ['header' => 'x'],
            '<?php

/*
 * x
 */

echo \'a\';',
            '<?php

/*
 * y
 * z
 */

echo \'a\';',
        ];

        yield [
            ['header' => "a\na"],
            '<?php

/*
 * a
 * a
 */

echo \'x\';',
            '<?php


/*
 * b
 * c
 */


echo \'x\';',
        ];

        yield [
            [
                'header' => 'foo',
                'location' => 'after_open',
                'separate' => 'bottom',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php
/**
 * foo
 */

declare(strict_types=1);

namespace A;

echo 1;',
            '<?php

declare(strict_types=1);
/**
 * foo
 */

namespace A;

echo 1;',
        ];

        yield [
            [
                'header' => 'foo',
                'location' => 'after_open',
                'separate' => 'bottom',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php
/**
 * foo
 */

declare(strict_types=1);
/**
 * bar
 */

namespace A;

echo 1;',
            '<?php

declare(strict_types=1);
/**
 * bar
 */

namespace A;

echo 1;',
        ];

        yield [
            [
                'header' => 'Foo',
                'separate' => 'none',
            ],
            '<?php

declare(strict_types=1);
/*
 * Foo
 */
namespace SebastianBergmann\Foo;

class Bar
{
}',
            '<?php
/*
 * Foo
 */

declare(strict_types=1);

namespace SebastianBergmann\Foo;

class Bar
{
}',
        ];

        yield [
            ['header' => 'tmp'],
            '<?php

/*
 * tmp
 */

/**
 * Foo class doc.
 */
class Foo {}',
            '<?php

/**
 * Foo class doc.
 */
class Foo {}',
        ];

        yield [
            ['header' => 'tmp'],
            '<?php

/*
 * tmp
 */

class Foo {}',
            '<?php

/*
 * Foo class doc.
 */
class Foo {}',
        ];

        yield [
            [
                'header' => 'tmp1',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * tmp1
 */

/**
 * Foo class doc.
 */
class Foo {}',
            '<?php

/**
 * Foo class doc.
 */
class Foo {}',
        ];

        yield [
            [
                'header' => 'tmp2',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * tmp2
 */

class Foo {}',
            '<?php

/**
 * tmp2
 */
class Foo {}',
        ];

        yield [
            [
                'header' => 'tmp3',
                'separate' => 'top',
            ],
            '<?php

/*
 * tmp3
 */
class Foo {}',
            '<?php
/**
 * Foo class doc.
 */
class Foo {}',
        ];

        yield [
            [
                'header' => 'bar',
                'location' => 'after_open',
            ],
            '<?php

/*
 * bar
 */

declare(strict_types=1);

// foo
foo();',
            '<?php

/*
 * foo
 */

declare(strict_types=1);

// foo
foo();',
        ];

        yield [
            [
                'header' => 'bar',
                'location' => 'after_open',
            ],
            '<?php

/*
 * bar
 */

declare(strict_types=1);

/* foo */
foo();',
            '<?php

/*
 * foo
 */

declare(strict_types=1);

/* foo */
foo();',
        ];

        yield [
            [
                'header' => 'tmp4',
                'location' => 'after_declare_strict',
            ],
            '<?php

/*
 * tmp4
 */

declare(strict_types=1) ?>',
            '<?php
declare(strict_types=1) ?>',
        ];

        yield [
            [
                'header' => 'tmp5',
                'location' => 'after_declare_strict',
            ],
            '#!/usr/bin/env php
<?php
declare(strict_types=1);

/*
 * tmp5
 */

namespace A\B;

echo 1;',
            '#!/usr/bin/env php
<?php
declare(strict_types=1);namespace A\B;

echo 1;',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_open',
            ],
            'Short mixed file A
Hello<?php echo "World!"; ?>',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_open',
            ],
            'Short mixed file B
<?php echo "Hello"; ?>World!',
        ];

        yield [
            [
                'header' => 'tmp',
                'location' => 'after_open',
            ],
            'File with anything at the beginning and with multiple opening tags are not supported
<?php
echo 1;
?>Hello World!<?php
script_continues_here();',
        ];

        $fileHeaderParts = [
            <<<'EOF'
                This file is part of the xxx.

                (c) Foo Bar <foo@bar.com>
                EOF,
            <<<'EOF'
                For the full copyright and license information, please view the LICENSE
                file that was distributed with this source code.
                EOF,
        ];
        $fileHeaderComment = implode('', $fileHeaderParts);
        $fileHeaderCommentValidator = implode('', [
            '/',
            preg_quote($fileHeaderParts[0], '/'),
            '(?P<EXTRA>.*)??',
            preg_quote($fileHeaderParts[1], '/'),
            '/s',
        ]);

        yield 'using validator, but existing comment in wrong place - adding new one' => [
            [
                'header' => $fileHeaderComment,
                'validator' => $fileHeaderCommentValidator,
                'location' => 'after_open',
            ],
            '<?php

/*
 * This file is part of the xxx.
 *
 * (c) Foo Bar <foo@bar.com>
 *
 * This
 * is
 * sub
 * note.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace A;

echo 1;',
            '<?php

declare(strict_types=1);
/*
 * This file is part of the xxx.
 *
 * (c) Foo Bar <foo@bar.com>
 *
 * This
 * is
 * sub
 * note.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace A;

echo 1;',
        ];

        yield 'using validator, existing comment matches' => [
            [
                'header' => $fileHeaderComment,
                'validator' => $fileHeaderCommentValidator,
                'location' => 'after_open',
            ],
            '<?php

/*
 * This file is part of the xxx.
 *
 * (c) Foo Bar <foo@bar.com>
 *
 * This
 * is
 * sub
 * note.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace A;

echo 1;',
        ];

        yield 'using validator, existing comment matches but misplaced' => [
            [
                'header' => $fileHeaderComment,
                'validator' => $fileHeaderCommentValidator,
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * This file is part of the xxx.
 *
 * (c) Foo Bar <foo@bar.com>
 *
 * This
 * is
 * sub
 * note.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

class Foo {}',
            '<?php

/**
 * This file is part of the xxx.
 *
 * (c) Foo Bar <foo@bar.com>
 *
 * This
 * is
 * sub
 * note.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
class Foo {}',
        ];

        yield 'default configuration' => [
            ['header' => 'a'],
            '<?php

/*
 * a
 */

echo 1;',
            '<?php
echo 1;',
        ];

        yield [
            [
                'header' => 'a',
                'comment_type' => HeaderCommentFixer::HEADER_COMMENT,
            ],
            '<?php

/*
 * a
 */

echo 1;',
            '<?php
echo 1;',
        ];

        yield [
            [
                'header' => 'a',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            '<?php

/**
 * a
 */

echo 1;',
            '<?php
echo 1;',
        ];

        yield [['header' => ''], "<?php\nphpinfo();\n?>\n<?"];

        yield [['header' => ''], " <?php\nphpinfo();\n"];

        yield [['header' => ''], "<?php\nphpinfo();\n?><hr/>"];

        yield [['header' => ''], "  <?php\n"];

        yield [['header' => ''], '<?= 1?>'];

        yield [['header' => ''], "<?= 1?><?php\n"];

        yield [['header' => ''], "<?= 1?>\n<?php\n"];

        yield [['header' => ''], "<?php\n// comment 1\n?><?php\n// comment 2\n"];

        yield [
            [
                'header' => 'whitemess',
                'location' => 'after_declare_strict',
                'separate' => 'bottom',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            "<?php\r\ndeclare(strict_types=1);\r\n/**\r\n * whitemess\r\n */\r\n\r\nnamespace A\\B;\r\n\r\necho 1;",
            "<?php\r\ndeclare(strict_types=1);\r\n\r\nnamespace A\\B;\r\n\r\necho 1;",
            "\t",
            "\r\n",
        ];

        yield [
            ['header' => 'Foo'],
            "<?php\n\n/*\n * Foo\n */\n\necho 1;",
            "<?php\necho 1;",
        ];

        yield [
            ['header' => 'Foo'],
            "<?php\r\n\r\n/*\r\n * Foo\r\n */\r\n\r\necho 1;",
            "<?php\r\necho 1;",
            '    ',
            "\r\n",
        ];

        yield [
            ['header' => 'Bar'],
            "<?php\r\n\r\n/*\r\n * Bar\r\n */\r\n\r\necho 1;",
            "<?php\r\necho 1;",
            '    ',
            "\r\n",
        ];

        yield [
            ['header' => 'Bar'],
            "<?php\n\n/*\n * Bar\n */\n\necho 1;",
            "<?php\necho 1;",
            '    ',
            "\n",
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(array $configuration, string $expected, ?string $input = null): void
    {
        $this->testFix($configuration, $expected, $input);
    }

    /**
     * @return iterable<int, array{0: _AutogeneratedInputConfiguration, 1: string, 2?: string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield [
            ['header' => 'tmp'],
            '<?php

/*
 * tmp
 */

/**
 * Foo class doc.
 */
enum Foo {}',
            '<?php

/**
 * Foo class doc.
 */
enum Foo {}',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration                 $configuration
     * @param class-string<InvalidFixerConfigurationException> $exception
     *
     * @dataProvider provideInvalidConfigurationCases
     */
    public function testInvalidConfiguration(
        ?array $configuration,
        string $exceptionMessage,
        string $exception = InvalidFixerConfigurationException::class
    ): void {
        $this->expectException($exception);
        $this->expectExceptionMessageMatches("#^\\[header_comment\\] {$exceptionMessage}$#");

        if (null !== $configuration) {
            $this->fixer->configure($configuration);
        } else {
            $this->doTest('<?php echo 1;');
        }
    }

    /**
     * @return iterable<int, array{null|array<array-key, mixed>, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield [
            null,
            'Configuration is required.',
            RequiredFixerConfigurationException::class,
        ];

        yield [[], 'Missing required configuration: The required option "header" is missing.'];

        yield [
            ['header' => 1],
            'Invalid configuration: The option "header" with value 1 is expected to be of type "string", but is of type "(int|integer)"\.',
        ];

        yield [
            [
                'header' => '',
                'comment_type' => 'foo',
            ],
            'Invalid configuration: The option "comment_type" with value "foo" is invalid\. Accepted values are: "PHPDoc", "comment"\.',
        ];

        yield [
            [
                'header' => '',
                'comment_type' => new \stdClass(),
            ],
            'Invalid configuration: The option "comment_type" with value stdClass is invalid\. Accepted values are: "PHPDoc", "comment"\.',
        ];

        yield [
            [
                'header' => '',
                'location' => new \stdClass(),
            ],
            'Invalid configuration: The option "location" with value stdClass is invalid\. Accepted values are: "after_open", "after_declare_strict"\.',
        ];

        yield [
            [
                'header' => '',
                'separate' => new \stdClass(),
            ],
            'Invalid configuration: The option "separate" with value stdClass is invalid\. Accepted values are: "both", "top", "bottom", "none"\.',
        ];

        yield [
            [
                'header' => 'Foo',
                'validator' => '/\w+++/',
            ],
            'Provided RegEx is not valid.',
        ];

        yield [
            [
                'header' => '/** test */',
                'comment_type' => HeaderCommentFixer::HEADER_PHPDOC,
            ],
            'Cannot use \'\*/\' in header\.$',
        ];
    }
}
