<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Comment;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Comment\SingleLineCommentStyleFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Comment\SingleLineCommentStyleFixer>
 *
 * @author Filippo Tessarotto <zoeslam@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Comment\SingleLineCommentStyleFixer
 */
final class SingleLineCommentStyleFixerTest extends AbstractFixerTestCase
{
    public function testInvalidConfiguration(): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);

        $this->fixer->configure(['abc']);
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php
// lonely line
',
            '<?php
/* lonely line */
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
   // indented line
',
            '<?php
   /* indented line */
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
   // weird-spaced line
',
            '<?php
   /*   weird-spaced line*/
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php // start-end',
            '<?php /* start-end */',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            "<?php\n \t \n \t // weird indent\n",
            "<?php\n \t \n \t /* weird indent */\n",
            ['comment_types' => ['asterisk']],
        ];

        yield [
            "<?php\n// with spaces after\n \t ",
            "<?php\n/* with spaces after */ \t \n \t ",
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
$a = 1; // after code
',
            '<?php
$a = 1; /* after code */
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
   /* first */ // second
',
            '<?php
   /* first */ /* second */
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
   /* first */// second',
            '<?php
   /* first *//*
   second
   */',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    // one line',
            '<?php
    /*one line

     */',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    // one line',
            '<?php
    /*

    one line*/',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    // one line',
            "<?php
    /* \t "."
 \t   * one line ".'
     *
     */',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
//',
            '<?php
/***
 *
 */',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php

    // s',
            '<?php

    /***
s    *
     */',
            ['comment_types' => ['asterisk']],
        ];

        yield 'empty comment' => [
            '<?php
//
',
            '<?php
/**/
',
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
$a = 1; /* in code */ $b = 2;
',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /*
     * in code 2
     */ $a = 1;
',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
/***
 *
 */ $a = 1;',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /***
s    *
     */ $a = 1;',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /*
     * first line
     * second line
     */',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /*
     * first line
     *
     * second line
     */',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /*first line
second line*/',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php /** inline doc comment */',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php
    /**
     * Doc comment
     */',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<?php # test',
            null,
            ['comment_types' => ['asterisk']],
        ];

        yield [
            '<h1>This is an <?php //echo 123;?> example</h1>',
            '<h1>This is an <?php #echo 123;?> example</h1>',
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php
                    // test
                ',
            '<?php
                    # test
                ',
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php
                    // test1
                    //test2
                    // test3
                    // test 4
                ',
            '<?php
                    # test1
                    #test2
                    # test3
                    # test 4
                ',
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php //',
            '<?php #',
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php
                    //#test
                ',
            null,
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php
                    /*
                        #test
                    */
                ',
            null,
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php // a',
            '<?php # a',
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php /* start-end */',
            null,
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php function foo(
    #[MyAttr([1, 2])] Type $myParam,
) {} // foo',
            null,
            ['comment_types' => ['hash']],
        ];

        yield [
            '<?php
    // 1
    // 2
    /*
     * 3.a
     * 3.b
     */
    /**
     * 4
     */
    // 5
',
            '<?php
    /* 1 */
    /*
     * 2
     */
    /*
     * 3.a
     * 3.b
     */
    /**
     * 4
     */
    # 5
',
        ];

        yield [
            '<?php
                function foo() {
                    /* ?> */
                    return "bar";
                }',
        ];
    }
}
