<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\DoctrineAnnotation;

use PhpCsFixer\Tests\AbstractDoctrineAnnotationFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\AbstractDoctrineAnnotationFixer
 * @covers \PhpCsFixer\Doctrine\Annotation\DocLexer
 * @covers \PhpCsFixer\Fixer\DoctrineAnnotation\DoctrineAnnotationSpacesFixer
 *
 * @extends AbstractDoctrineAnnotationFixerTestCase<\PhpCsFixer\Fixer\DoctrineAnnotation\DoctrineAnnotationSpacesFixer>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\DoctrineAnnotation\DoctrineAnnotationSpacesFixer
 */
final class DoctrineAnnotationSpacesFixerTest extends AbstractDoctrineAnnotationFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo
 */'],
                ['
/**
 * @Foo()
 */'],
                ['
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo(foo="foo", bar="bar")
 */', '
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo ( foo = "foo" ,bar = "bar" )
 */'],
                ['
/**
 * @Foo(
 *     foo="foo",
 *     bar="bar"
 * )
 */', '
/**
 * @Foo (
 *     foo = "foo" ,
 *     bar = "bar"
 * )
 */'],
                ['
/**
 * @Foo(
 *     @Bar("foo", "bar"),
 *     @Baz
 * )
 */', '
/**
 * @Foo(
 *     @Bar ( "foo" ,"bar") ,
 *     @Baz
 * )
 */'],
                ['
/**
 * @Foo({"bar", "baz"})
 */', '
/**
 * @Foo( {"bar" ,"baz"} )
 */'],
                ['
/**
 * @Foo(foo="=foo", bar={"foo" : "=foo", "bar" = "=bar"})
 */', '
/**
 * @Foo(foo = "=foo" ,bar = {"foo" : "=foo", "bar"="=bar"})
 */'],
                [
                    '/** @Foo(foo="foo", bar={"foo" : "foo", "bar" = "bar"}) */',
                    '/** @Foo ( foo = "foo" ,bar = {"foo" : "foo", "bar"="bar"} ) */',
                ],
                ['
/**
 * @Foo(
 *     foo="foo",
 *     bar={
 *         "foo" : "foo",
 *         "bar" = "bar"
 *     }
 * )
 */', '
/**
 * @Foo(
 *     foo = "foo"
 *     ,
 *     bar = {
 *         "foo":"foo",
 *         "bar"="bar"
 *     }
 * )
 */'],
                ['
/**
 * @Foo(
 *     foo="foo",
 *     bar={
 *         "foo" : "foo",
 *         "bar" = "bar"
 *     }
 * )
 */', '
/**
 * @Foo
 * (
 *     foo
 *     =
 *     "foo",
 *     bar
 *     =
 *     {
 *         "foo"
 *         :
 *         "foo",
 *         "bar"
 *         =
 *         "bar"
 *     }
 * )
 */'],
                ['
/**
 * @Foo(foo="foo", "bar"=@Bar\Baz({"foo" : true, "bar" = false}))
 */', '
/**
 * @Foo   (   foo = "foo", "bar" = @Bar\Baz({"foo":true, "bar"=false})   )
 */'],
                ['
/**
 * @Foo(foo = "foo" ,bar="bar"
 */'],
                ['
/**
 * Comment , with a comma.
 */'],
                ['
/**
 * Description with a single " character.
 *
 * @Foo(foo="string "" with inner quote", bar="string "" with inner quote")
 *
 * @param mixed description with a single " character.
 */', '
/**
 * Description with a single " character.
 *
 * @Foo( foo="string "" with inner quote" ,bar="string "" with inner quote" )
 *
 * @param mixed description with a single " character.
 */'],
                ['
/**
 * // PHPDocumentor 1
 * @abstract ( foo,bar  =  "baz" )
 * @access ( foo,bar  =  "baz" )
 * @code ( foo,bar  =  "baz" )
 * @deprec ( foo,bar  =  "baz" )
 * @encode ( foo,bar  =  "baz" )
 * @exception ( foo,bar  =  "baz" )
 * @final ( foo,bar  =  "baz" )
 * @ingroup ( foo,bar  =  "baz" )
 * @inheritdoc ( foo,bar  =  "baz" )
 * @inheritDoc ( foo,bar  =  "baz" )
 * @magic ( foo,bar  =  "baz" )
 * @name ( foo,bar  =  "baz" )
 * @toc ( foo,bar  =  "baz" )
 * @tutorial ( foo,bar  =  "baz" )
 * @private ( foo,bar  =  "baz" )
 * @static ( foo,bar  =  "baz" )
 * @staticvar ( foo,bar  =  "baz" )
 * @staticVar ( foo,bar  =  "baz" )
 * @throw ( foo,bar  =  "baz" )
 *
 * // PHPDocumentor 2
 * @api ( foo,bar  =  "baz" )
 * @author ( foo,bar  =  "baz" )
 * @category ( foo,bar  =  "baz" )
 * @copyright ( foo,bar  =  "baz" )
 * @deprecated ( foo,bar  =  "baz" )
 * @example ( foo,bar  =  "baz" )
 * @filesource ( foo,bar  =  "baz" )
 * @global ( foo,bar  =  "baz" )
 * @ignore ( foo,bar  =  "baz" )
 * @internal ( foo,bar  =  "baz" )
 * @license ( foo,bar  =  "baz" )
 * @link ( foo,bar  =  "baz" )
 * @method ( foo,bar  =  "baz" )
 * @package ( foo,bar  =  "baz" )
 * @param ( foo,bar  =  "baz" )
 * @property ( foo,bar  =  "baz" )
 * @property-read ( foo,bar  =  "baz" )
 * @property-write ( foo,bar  =  "baz" )
 * @return ( foo,bar  =  "baz" )
 * @see ( foo,bar  =  "baz" )
 * @since ( foo,bar  =  "baz" )
 * @source ( foo,bar  =  "baz" )
 * @subpackage ( foo,bar  =  "baz" )
 * @throws ( foo,bar  =  "baz" )
 * @todo ( foo,bar  =  "baz" )
 * @TODO ( foo,bar  =  "baz" )
 * @usedBy ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 * @var ( foo,bar  =  "baz" )
 * @version ( foo,bar  =  "baz" )
 *
 * // PHPUnit
 * @after ( foo,bar  =  "baz" )
 * @afterClass ( foo,bar  =  "baz" )
 * @backupGlobals ( foo,bar  =  "baz" )
 * @backupStaticAttributes ( foo,bar  =  "baz" )
 * @before ( foo,bar  =  "baz" )
 * @beforeClass ( foo,bar  =  "baz" )
 * @codeCoverageIgnore ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreStart ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreEnd ( foo,bar  =  "baz" )
 * @covers ( foo,bar  =  "baz" )
 * @coversDefaultClass ( foo,bar  =  "baz" )
 * @coversNothing ( foo,bar  =  "baz" )
 * @dataProvider ( foo,bar  =  "baz" )
 * @depends ( foo,bar  =  "baz" )
 * @expectedException ( foo,bar  =  "baz" )
 * @expectedExceptionCode ( foo,bar  =  "baz" )
 * @expectedExceptionMessage ( foo,bar  =  "baz" )
 * @expectedExceptionMessageRegExp ( foo,bar  =  "baz" )
 * @group ( foo,bar  =  "baz" )
 * @large ( foo,bar  =  "baz" )
 * @medium ( foo,bar  =  "baz" )
 * @preserveGlobalState ( foo,bar  =  "baz" )
 * @requires ( foo,bar  =  "baz" )
 * @runTestsInSeparateProcesses ( foo,bar  =  "baz" )
 * @runInSeparateProcess ( foo,bar  =  "baz" )
 * @small ( foo,bar  =  "baz" )
 * @test ( foo,bar  =  "baz" )
 * @testdox ( foo,bar  =  "baz" )
 * @ticket ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 *
 * // PHPCheckStyle
 * @SuppressWarnings ( foo,bar  =  "baz" )
 *
 * // PHPStorm
 * @noinspection ( foo,bar  =  "baz" )
 *
 * // PEAR
 * @package_version ( foo,bar  =  "baz" )
 *
 * // PlantUML
 * @enduml ( foo,bar  =  "baz" )
 * @startuml ( foo,bar  =  "baz" )
 *
 * // other
 * @fix ( foo,bar  =  "baz" )
 * @FIXME ( foo,bar  =  "baz" )
 * @fixme ( foo,bar  =  "baz" )
 * @override
 */'],
                ['
/**
 * @Transform /^(\d+)$/

 */'],
            ],
            [
                [],
                [
                    'around_parentheses' => true,
                    'around_commas' => true,
                    'before_argument_assignments' => false,
                    'after_argument_assignments' => false,
                    'before_array_assignments_equals' => true,
                    'after_array_assignments_equals' => true,
                    'before_array_assignments_colon' => true,
                    'after_array_assignments_colon' => true,
                ],
            ]
        );

        yield [
            '<?php

/**
 * @see \User getId()
 */
',
        ];

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo
 */'],
                ['
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo()
 */', '
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo ( )
 */'],
                ['
/**
 * @Foo("bar")
 */', '
/**
 * @Foo( "bar" )
 */'],
                ['
/**
 * @Foo("bar", "baz")
 */', '
/**
 * @Foo( "bar", "baz" )
 */'],
                [
                    '/** @Foo("bar", "baz") */',
                    '/** @Foo( "bar", "baz" ) */',
                ],
                ['
/**
 * @Foo("bar", "baz")
 */', '
/**
 * @Foo(     "bar", "baz"     )
 */'],
                ['
/**
 * @Foo("bar", "baz")
 */', '
/**
 * @Foo    (     "bar", "baz"     )
 */'],
                ['
/**
 * @Foo(
 *     "bar",
 *     "baz"
 * )
 */', '
/**
 * @Foo
 * (
 *     "bar",
 *     "baz"
 * )
 */'],
                ['
/**
 * @Foo(
 *     @Bar("baz")
 * )
 */', '
/**
 * @Foo
 * (
 *     @Bar ( "baz" )
 * )
 */'],
                ['
/**
 * @Foo ( @Bar ( "bar" )
 */'],
                ['
/**
 * Foo ( Bar Baz )
 */'],
                ['
/**
 * Description with a single " character.
 *
 * @Foo("string "" with inner quote")
 *
 * @param mixed description with a single " character.
 */', '
/**
 * Description with a single " character.
 *
 * @Foo ( "string "" with inner quote" )
 *
 * @param mixed description with a single " character.
 */'],
                ['
/**
 * // PHPDocumentor 1
 * @abstract ( foo,bar  =  "baz" )
 * @access ( foo,bar  =  "baz" )
 * @code ( foo,bar  =  "baz" )
 * @deprec ( foo,bar  =  "baz" )
 * @encode ( foo,bar  =  "baz" )
 * @exception ( foo,bar  =  "baz" )
 * @final ( foo,bar  =  "baz" )
 * @ingroup ( foo,bar  =  "baz" )
 * @inheritdoc ( foo,bar  =  "baz" )
 * @inheritDoc ( foo,bar  =  "baz" )
 * @magic ( foo,bar  =  "baz" )
 * @name ( foo,bar  =  "baz" )
 * @toc ( foo,bar  =  "baz" )
 * @tutorial ( foo,bar  =  "baz" )
 * @private ( foo,bar  =  "baz" )
 * @static ( foo,bar  =  "baz" )
 * @staticvar ( foo,bar  =  "baz" )
 * @staticVar ( foo,bar  =  "baz" )
 * @throw ( foo,bar  =  "baz" )
 *
 * // PHPDocumentor 2
 * @api ( foo,bar  =  "baz" )
 * @author ( foo,bar  =  "baz" )
 * @category ( foo,bar  =  "baz" )
 * @copyright ( foo,bar  =  "baz" )
 * @deprecated ( foo,bar  =  "baz" )
 * @example ( foo,bar  =  "baz" )
 * @filesource ( foo,bar  =  "baz" )
 * @global ( foo,bar  =  "baz" )
 * @ignore ( foo,bar  =  "baz" )
 * @internal ( foo,bar  =  "baz" )
 * @license ( foo,bar  =  "baz" )
 * @link ( foo,bar  =  "baz" )
 * @method ( foo,bar  =  "baz" )
 * @package ( foo,bar  =  "baz" )
 * @param ( foo,bar  =  "baz" )
 * @property ( foo,bar  =  "baz" )
 * @property-read ( foo,bar  =  "baz" )
 * @property-write ( foo,bar  =  "baz" )
 * @return ( foo,bar  =  "baz" )
 * @see ( foo,bar  =  "baz" )
 * @since ( foo,bar  =  "baz" )
 * @source ( foo,bar  =  "baz" )
 * @subpackage ( foo,bar  =  "baz" )
 * @throws ( foo,bar  =  "baz" )
 * @todo ( foo,bar  =  "baz" )
 * @TODO ( foo,bar  =  "baz" )
 * @usedBy ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 * @var ( foo,bar  =  "baz" )
 * @version ( foo,bar  =  "baz" )
 *
 * // PHPUnit
 * @after ( foo,bar  =  "baz" )
 * @afterClass ( foo,bar  =  "baz" )
 * @backupGlobals ( foo,bar  =  "baz" )
 * @backupStaticAttributes ( foo,bar  =  "baz" )
 * @before ( foo,bar  =  "baz" )
 * @beforeClass ( foo,bar  =  "baz" )
 * @codeCoverageIgnore ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreStart ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreEnd ( foo,bar  =  "baz" )
 * @covers ( foo,bar  =  "baz" )
 * @coversDefaultClass ( foo,bar  =  "baz" )
 * @coversNothing ( foo,bar  =  "baz" )
 * @dataProvider ( foo,bar  =  "baz" )
 * @depends ( foo,bar  =  "baz" )
 * @expectedException ( foo,bar  =  "baz" )
 * @expectedExceptionCode ( foo,bar  =  "baz" )
 * @expectedExceptionMessage ( foo,bar  =  "baz" )
 * @expectedExceptionMessageRegExp ( foo,bar  =  "baz" )
 * @group ( foo,bar  =  "baz" )
 * @large ( foo,bar  =  "baz" )
 * @medium ( foo,bar  =  "baz" )
 * @preserveGlobalState ( foo,bar  =  "baz" )
 * @requires ( foo,bar  =  "baz" )
 * @runTestsInSeparateProcesses ( foo,bar  =  "baz" )
 * @runInSeparateProcess ( foo,bar  =  "baz" )
 * @small ( foo,bar  =  "baz" )
 * @test ( foo,bar  =  "baz" )
 * @testdox ( foo,bar  =  "baz" )
 * @ticket ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 *
 * // PHPCheckStyle
 * @SuppressWarnings ( foo,bar  =  "baz" )
 *
 * // PHPStorm
 * @noinspection ( foo,bar  =  "baz" )
 *
 * // PEAR
 * @package_version ( foo,bar  =  "baz" )
 *
 * // PlantUML
 * @enduml ( foo,bar  =  "baz" )
 * @startuml ( foo,bar  =  "baz" )
 *
 * // other
 * @fix ( foo,bar  =  "baz" )
 * @FIXME ( foo,bar  =  "baz" )
 * @fixme ( foo,bar  =  "baz" )
 * @override
 */'],
            ],
            [
                [
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
                [
                    'around_parentheses' => true,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo
 */'],
                ['
/**
 * @Foo()
 */'],
                ['
/**
 * @Foo ()
 */'],
                ['
/**
 * @Foo( "bar" )
 */'],
                ['
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo( "bar", "baz")
 */', '
/**
 * Foo.
 *
 * @author John Doe
 *
 * @Foo( "bar" ,"baz")
 */'],
                [
                    '/** @Foo( "bar", "baz") */',
                    '/** @Foo( "bar" ,"baz") */',
                ],
                ['
/**
 * @Foo( "bar", "baz")
 */', '
/**
 * @Foo( "bar" , "baz")
 */'],
                ['
/**
 * @Foo(
 *     "bar",
 *     "baz"
 * )
 */', '
/**
 * @Foo(
 *     "bar" ,
 *     "baz"
 * )
 */'],
                ['
/**
 * @Foo(
 *     "bar",
 *     "baz"
 * )
 */', '
/**
 * @Foo(
 *     "bar"
 *     ,
 *     "baz"
 * )
 */'],
                ['
/**
 * @Foo("bar ,", "baz,")
 */'],
                ['
/**
 * @Foo(
 *     @Bar ( "foo", "bar"),
 *     @Baz
 * )
 */', '
/**
 * @Foo(
 *     @Bar ( "foo" ,"bar") ,
 *     @Baz
 * )
 */'],
                ['
/**
 * @Foo({"bar", "baz"})
 */', '
/**
 * @Foo({"bar" ,"baz"})
 */'],
                ['
/**
 * @Foo(foo="foo", bar="bar")
 */', '
/**
 * @Foo(foo="foo" ,bar="bar")
 */'],
                ['
/**
 * @Foo(foo="foo" ,bar="bar"
 */'],
                ['
/**
 * Comment , with a comma.
 */'],
                ['
/**
 * Description with a single " character.
 *
 * @Foo(foo="string "" with inner quote", bar="string "" with inner quote")
 *
 * @param mixed description with a single " character.
 */', '
/**
 * Description with a single " character.
 *
 * @Foo(foo="string "" with inner quote" ,bar="string "" with inner quote")
 *
 * @param mixed description with a single " character.
 */'],
                ['
/**
 * // PHPDocumentor 1
 * @abstract ( foo,bar  =  "baz" )
 * @access ( foo,bar  =  "baz" )
 * @code ( foo,bar  =  "baz" )
 * @deprec ( foo,bar  =  "baz" )
 * @encode ( foo,bar  =  "baz" )
 * @exception ( foo,bar  =  "baz" )
 * @final ( foo,bar  =  "baz" )
 * @ingroup ( foo,bar  =  "baz" )
 * @inheritdoc ( foo,bar  =  "baz" )
 * @inheritDoc ( foo,bar  =  "baz" )
 * @magic ( foo,bar  =  "baz" )
 * @name ( foo,bar  =  "baz" )
 * @toc ( foo,bar  =  "baz" )
 * @tutorial ( foo,bar  =  "baz" )
 * @private ( foo,bar  =  "baz" )
 * @static ( foo,bar  =  "baz" )
 * @staticvar ( foo,bar  =  "baz" )
 * @staticVar ( foo,bar  =  "baz" )
 * @throw ( foo,bar  =  "baz" )
 *
 * // PHPDocumentor 2
 * @api ( foo,bar  =  "baz" )
 * @author ( foo,bar  =  "baz" )
 * @category ( foo,bar  =  "baz" )
 * @copyright ( foo,bar  =  "baz" )
 * @deprecated ( foo,bar  =  "baz" )
 * @example ( foo,bar  =  "baz" )
 * @filesource ( foo,bar  =  "baz" )
 * @global ( foo,bar  =  "baz" )
 * @ignore ( foo,bar  =  "baz" )
 * @internal ( foo,bar  =  "baz" )
 * @license ( foo,bar  =  "baz" )
 * @link ( foo,bar  =  "baz" )
 * @method ( foo,bar  =  "baz" )
 * @package ( foo,bar  =  "baz" )
 * @param ( foo,bar  =  "baz" )
 * @property ( foo,bar  =  "baz" )
 * @property-read ( foo,bar  =  "baz" )
 * @property-write ( foo,bar  =  "baz" )
 * @return ( foo,bar  =  "baz" )
 * @see ( foo,bar  =  "baz" )
 * @since ( foo,bar  =  "baz" )
 * @source ( foo,bar  =  "baz" )
 * @subpackage ( foo,bar  =  "baz" )
 * @throws ( foo,bar  =  "baz" )
 * @todo ( foo,bar  =  "baz" )
 * @TODO ( foo,bar  =  "baz" )
 * @usedBy ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 * @var ( foo,bar  =  "baz" )
 * @version ( foo,bar  =  "baz" )
 *
 * // PHPUnit
 * @after ( foo,bar  =  "baz" )
 * @afterClass ( foo,bar  =  "baz" )
 * @backupGlobals ( foo,bar  =  "baz" )
 * @backupStaticAttributes ( foo,bar  =  "baz" )
 * @before ( foo,bar  =  "baz" )
 * @beforeClass ( foo,bar  =  "baz" )
 * @codeCoverageIgnore ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreStart ( foo,bar  =  "baz" )
 * @codeCoverageIgnoreEnd ( foo,bar  =  "baz" )
 * @covers ( foo,bar  =  "baz" )
 * @coversDefaultClass ( foo,bar  =  "baz" )
 * @coversNothing ( foo,bar  =  "baz" )
 * @dataProvider ( foo,bar  =  "baz" )
 * @depends ( foo,bar  =  "baz" )
 * @expectedException ( foo,bar  =  "baz" )
 * @expectedExceptionCode ( foo,bar  =  "baz" )
 * @expectedExceptionMessage ( foo,bar  =  "baz" )
 * @expectedExceptionMessageRegExp ( foo,bar  =  "baz" )
 * @group ( foo,bar  =  "baz" )
 * @large ( foo,bar  =  "baz" )
 * @medium ( foo,bar  =  "baz" )
 * @preserveGlobalState ( foo,bar  =  "baz" )
 * @requires ( foo,bar  =  "baz" )
 * @runTestsInSeparateProcesses ( foo,bar  =  "baz" )
 * @runInSeparateProcess ( foo,bar  =  "baz" )
 * @small ( foo,bar  =  "baz" )
 * @test ( foo,bar  =  "baz" )
 * @testdox ( foo,bar  =  "baz" )
 * @ticket ( foo,bar  =  "baz" )
 * @uses ( foo,bar  =  "baz" )
 *
 * // PHPCheckStyle
 * @SuppressWarnings ( foo,bar  =  "baz" )
 *
 * // PHPStorm
 * @noinspection ( foo,bar  =  "baz" )
 *
 * // PEAR
 * @package_version ( foo,bar  =  "baz" )
 *
 * // PlantUML
 * @enduml ( foo,bar  =  "baz" )
 * @startuml ( foo,bar  =  "baz" )
 *
 * // other
 * @fix ( foo,bar  =  "baz" )
 * @FIXME ( foo,bar  =  "baz" )
 * @fixme ( foo,bar  =  "baz" )
 * @override
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
                [
                    'around_parentheses' => false,
                    'around_commas' => true,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo ="foo", bar ={"foo":"foo", "bar"="bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => true,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo= "foo", bar= {"foo" : "foo", "bar" = "bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => false,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo= "foo", bar= {"foo":"foo", "bar"="bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => true,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo ="foo", bar ={"foo" : "foo", "bar" = "bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => false,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar" ="bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => true,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar"= "bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => false,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"= "bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => true,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" ="bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => false,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo" :"foo", "bar"="bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => true,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo": "foo", "bar" = "bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => false,
                    'after_array_assignments_colon' => null,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo": "foo", "bar"="bar"})
 */', '
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => true,
                ],
            ]
        );

        yield from self::createMultipleTestCase(
            [
                ['
/**
 * @Foo(foo="foo", bar={"foo":"foo", "bar"="bar"})
 */'],
                ['
/**
 * @Foo(foo = "foo", bar = {"foo" :"foo", "bar" = "bar"})
 */', '
/**
 * @Foo(foo = "foo", bar = {"foo" : "foo", "bar" = "bar"})
 */'],
            ],
            [
                [
                    'around_parentheses' => false,
                    'around_commas' => false,
                    'before_argument_assignments' => null,
                    'after_argument_assignments' => null,
                    'before_array_assignments_equals' => null,
                    'after_array_assignments_equals' => null,
                    'before_array_assignments_colon' => null,
                    'after_array_assignments_colon' => false,
                ],
            ]
        );

        $elements = [
            'private $foo;',
            'private string $foo;',
            'private Foo\Bar $foo;',
            'private ?Foo\Bar $foo;',
        ];

        foreach ($elements as $element) {
            yield [
                \sprintf('<?php
                class Foo
                {
                    /**
                     * @Foo(foo="foo")
                     */
                    %s
                }
            ', $element),
                \sprintf('<?php
                class Foo
                {
                    /**
                     * @Foo(foo = "foo")
                     */
                    %s
                }
            ', $element),
            ];
        }
    }

    /**
     * @param list<array{0: string, 1?: string}>     $commentCases
     * @param list<_AutogeneratedInputConfiguration> $configurations
     *
     * @return iterable<array{string, null|string, _AutogeneratedInputConfiguration}>
     */
    private static function createMultipleTestCase(array $commentCases, array $configurations): iterable
    {
        foreach ($configurations as $configuration) {
            foreach (self::createTestCases($commentCases, $configuration) as $testCase) {
                yield $testCase;

                $testCase[0] = str_replace("\n", "\r\n", $testCase[0]);
                if (null !== $testCase[1]) {
                    $testCase[1] = str_replace("\n", "\r\n", $testCase[1]);
                }

                yield $testCase;
            }
        }
    }
}
