<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Import;

use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Import\GlobalNamespaceImportFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Import\GlobalNamespaceImportFixer>
 *
 * @author Gregor Harlan <gharlan@web.de>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Import\GlobalNamespaceImportFixer
 */
final class GlobalNamespaceImportFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{string, null|string, _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield 'import_constants (true) - non-global names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                echo FOO, \Bar\BAZ, namespace\FOO2;
                EXPECTED,
            null,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - name already used [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                echo \FOO, FOO, \FOO;
                EXPECTED,
            null,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - name already used [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const Bar\FOO;
                echo \FOO;
                EXPECTED,
            null,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - name already used [3]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                const FOO = 1;
                echo \FOO;
                EXPECTED,
            null,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - without namespace / do not import' => [
            <<<'INPUT'
                <?php
                echo \FOO, \BAR, \FOO;
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - with namespace' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const BAR;
                use const FOO;
                echo FOO, BAR;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                echo \FOO, \BAR;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - with namespace with {} syntax' => [
            <<<'EXPECTED'
                <?php
                namespace Test {
                use const BAR;
                use const FOO;
                    echo FOO, BAR;
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test {
                    echo \FOO, \BAR;
                }
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - ignore other imported types' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use BAR;
                use const BAR;
                use const FOO;
                echo FOO, BAR;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use BAR;
                echo \FOO, \BAR;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - respect already imported names [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const BAR;
                use const FOO;
                echo FOO, BAR;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use const BAR;
                echo \FOO, \BAR;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - respect already imported names [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const \BAR;
                use const FOO;
                echo FOO, BAR, BAR;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use const \BAR;
                echo \FOO, \BAR, BAR;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - handle case sensitivity' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const fOO;
                use const FOO;
                use const Foo;
                const foO = 1;
                echo FOO, Foo;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use const fOO;
                const foO = 1;
                echo \FOO, \Foo;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - handle aliased imports' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const BAR as BAZ;
                use const FOO;
                echo FOO, BAZ;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use const BAR as BAZ;
                echo \FOO, \BAR;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - ignore class constants' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const FOO;
                class Bar {
                    const FOO = 1;
                }
                echo FOO;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                class Bar {
                    const FOO = 1;
                }
                echo \FOO;
                INPUT,
            ['import_constants' => true],
        ];

        yield 'import_constants (true) - global namespace' => [
            <<<'INPUT'
                <?php
                echo \FOO, \BAR;
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield [
            <<<'INPUT'
                <?php
                namespace {
                    echo \FOO, \BAR;
                }
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield 'import_functions (true) - non-global names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                foo();
                Bar\baz();
                namespace\foo2();
                EXPECTED,
            null,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - name already used [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                \foo();
                Foo();
                \foo();
                EXPECTED,
            null,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - name already used [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function Bar\foo;
                \Foo();
                EXPECTED,
            null,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - name already used [3]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                function foo() {}
                \Foo();
                EXPECTED,
            null,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - without namespace / do not import' => [
            <<<'INPUT'
                <?php
                \foo();
                \bar();
                \Foo();
                INPUT,
            null,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - with namespace' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function bar;
                use function foo;
                foo();
                bar();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                \foo();
                \bar();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - with namespace with {} syntax' => [
            <<<'EXPECTED'
                <?php
                namespace Test {
                use function bar;
                use function foo;
                    foo();
                    bar();
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test {
                    \foo();
                    \bar();
                }
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - ignore other imported types' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use bar;
                use function bar;
                use function foo;
                foo();
                bar();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use bar;
                \foo();
                \bar();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - respect already imported names [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function bar;
                use function foo;
                foo();
                Bar();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use function bar;
                \foo();
                \Bar();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - respect already imported names [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function \bar;
                use function foo;
                foo();
                Bar();
                bar();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use function \bar;
                \foo();
                \Bar();
                bar();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - handle aliased imports' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function bar as baz;
                use function foo;
                foo();
                baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use function bar as baz;
                \foo();
                \Bar();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - ignore class methods' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function foo;
                class Bar {
                    function foo() {}
                }
                foo();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                class Bar {
                    function foo() {}
                }
                \foo();
                INPUT,
            ['import_functions' => true],
        ];

        yield 'import_functions (true) - name already used' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                class Bar {
                    function baz() {
                        new class() {
                            function baz() {
                                function foo() {}
                            }
                        };
                    }
                }
                \foo();
                EXPECTED,
            null,
            ['import_functions' => true],
        ];

        yield 'import_classes (true) - non-global names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                new Foo();
                new Bar\Baz();
                new namespace\Foo2();

                /** @var Foo|Bar\Baz $x */
                $x = x();
                EXPECTED,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - name already used [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                new \Foo();
                new foo();

                /** @var \Foo $foo */
                $foo = new \Foo();
                EXPECTED,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - name already used [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar\foo;

                /** @var \Foo $foo */
                $foo = new \Foo();
                EXPECTED,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - name already used [3]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                class foo {}

                /** @var \Foo $foo */
                $foo = new \Foo();
                EXPECTED,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - name already used [4]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;

                /** @return array<string, foo> */
                function x() {}

                /** @var \Foo $foo */
                $foo = new \Foo();
                EXPECTED,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - without namespace / do not import' => [
            <<<'INPUT'
                <?php
                /** @var \Foo $foo */
                $foo = new \foo();
                new \Bar();
                \FOO::baz();
                INPUT,
            null,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - with namespace' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar;
                use Baz;
                use Foo;

                new Foo();
                Bar::baz();

                /** @return Baz<string, foo> */
                function x() {}
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;

                new \Foo();
                \Bar::baz();

                /** @return \Baz<string, \foo> */
                function x() {}
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - with namespace with {} syntax' => [
            <<<'EXPECTED'
                <?php
                namespace Test {
                use Bar;
                use Foo;
                    new Foo();
                    Bar::baz();
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test {
                    new \Foo();
                    \Bar::baz();
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - phpdoc only' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Throwable;

                /** @throws Throwable */
                function x() {}
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;

                /** @throws \Throwable */
                function x() {}
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - ignore other imported types' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function Bar;
                use Bar;
                use Foo;
                new Foo();
                Bar::baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use function Bar;
                new \Foo();
                \Bar::baz();
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - respect already imported names [1]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar;
                use Foo;
                new Foo();
                bar::baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Bar;
                new \Foo();
                \bar::baz();
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - respect already imported names [2]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use \Bar;
                use Foo;
                new Foo();
                new bar();
                new Bar();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use \Bar;
                new \Foo();
                new \bar();
                new Bar();
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - respect already imported names [3]' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Throwable;

                /** @throws Throwable */
                function x() {}

                /** @throws Throwable */
                function y() {}
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Throwable;

                /** @throws Throwable */
                function x() {}

                /** @throws \Throwable */
                function y() {}
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - handle aliased imports' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar as Baz;
                use Foo;

                new Foo();

                /** @var Baz $bar */
                $bar = new Baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Bar as Baz;

                new \Foo();

                /** @var \bar $bar */
                $bar = new \bar();
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - handle typehints' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar;
                use Baz;
                use Foo;
                class Abc {
                    function bar(Foo $a, Bar $b, foo &$c, Baz ...$d) {}
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                class Abc {
                    function bar(\Foo $a, \Bar $b, \foo &$c, \Baz ...$d) {}
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - handle typehints 2' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Bar;
                use Foo;
                class Abc {
                    function bar(?Foo $a): ?Bar {}
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                class Abc {
                    function bar(?\Foo $a): ?\Bar {}
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - try catch' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (Exception $e) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                try {
                } catch (\Exception $e) {
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - try catch with comments' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* ... */ Exception $e /* ... */) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                try {
                } catch (/* ... */ \Exception $e /* ... */) {
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_constants (false) - already fqn or sub namespace' => [
            <<<'EXPECTED'
                <?php
                use const FOO;
                use const BAR;
                echo \FOO, Baz\BAR;
                EXPECTED,
            null,
            ['import_constants' => false],
        ];

        yield 'import_constants (false) - handle all occurrences' => [
            <<<'EXPECTED'
                <?php
                namespace X;
                use const FOO;
                use const BAR;
                echo \FOO, \BAR, \FOO;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace X;
                use const FOO;
                use const BAR;
                echo FOO, BAR, FOO;
                INPUT,
            ['import_constants' => false],
        ];

        yield 'import_constants (false) - ignore other imports and non-imported names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use FOO;
                use const BAR;
                use const Baz;
                echo FOO, \BAR, BAZ, QUX;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use FOO;
                use const BAR;
                use const Baz;
                echo FOO, BAR, BAZ, QUX;
                INPUT,
            ['import_constants' => false],
        ];

        yield 'import_functions (false) - already fqn or sub namespace' => [
            <<<'EXPECTED'
                <?php
                use function foo;
                use function bar;
                \foo();
                Baz\bar();
                EXPECTED,
            null,
            ['import_functions' => false],
        ];

        yield 'import_functions (false) - handle all occurrences' => [
            <<<'EXPECTED'
                <?php
                namespace X;
                use function foo;
                use function bar;
                \foo();
                \bar();
                \Foo();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace X;
                use function foo;
                use function bar;
                foo();
                bar();
                Foo();
                INPUT,
            ['import_functions' => false],
        ];

        yield 'import_functions (false) - ignore other imports and non-imported names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use foo;
                use function bar;
                foo();
                \bar();
                baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use foo;
                use function bar;
                foo();
                bar();
                baz();
                INPUT,
            ['import_functions' => false],
        ];

        yield 'import_classes (false) - already fqn or sub namespace' => [
            <<<'EXPECTED'
                <?php
                use Foo;
                use Bar;

                new \Foo();
                Baz\Bar::baz();

                /**
                 * @param \Foo $foo
                 * @param Baz\Bar $bar
                 */
                function abc(\Foo $foo, Baz\Bar $bar = null) {}
                EXPECTED,
            null,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - handle all occurrences' => [
            <<<'EXPECTED'
                <?php
                namespace X;
                use Foo;
                use Bar;
                use IteratorAggregate;

                new \Foo();
                new \Bar();
                \foo::baz();

                /**
                 * @param \Foo|string $foo
                 * @param null|\Bar[] $bar
                 * @param-out \Foo $foo
                 * @return array<string, ?\Bar<int, \foo>>|null
                 */
                function abc($foo, \Bar $bar = null) {}

                /**
                 * @extends \Foo<\Bar>
                 * @implements \IteratorAggregate<\Foo>
                 */
                class Qux implements \IteratorAggregate {}
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace X;
                use Foo;
                use Bar;
                use IteratorAggregate;

                new Foo();
                new Bar();
                foo::baz();

                /**
                 * @param Foo|string $foo
                 * @param null|Bar[] $bar
                 * @param-out Foo $foo
                 * @return array<string, ?Bar<int, foo>>|null
                 */
                function abc($foo, Bar $bar = null) {}

                /**
                 * @extends Foo<Bar>
                 * @implements IteratorAggregate<Foo>
                 */
                class Qux implements IteratorAggregate {}
                INPUT,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - ignore other imports and non-imported names' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function Foo;
                use Bar;
                new Foo();
                new \Bar();
                new Baz();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use function Foo;
                use Bar;
                new Foo();
                new Bar();
                new Baz();
                INPUT,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - try catch' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (\Exception $e) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (Exception $e) {
                }
                INPUT,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - try catch with comments' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* ... */ \Exception $e /* ... */) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* ... */ Exception $e /* ... */) {
                }
                INPUT,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - key in PHPDoc\'s array shape matching class name' => [
            '<?php
                namespace Foo;
                use Exception;
                class Bar {
                    /**
                     * @return array{code: int, exception: \Exception}
                     */
                    public function f1(): array {}
                    /**
                     * @return array{exception: \Exception}
                     */
                    public function f2(): array {}
                    /**
                     * @return array{exceptions: array<\Exception>}
                     */
                    public function f3(): array {}
                }',
            '<?php
                namespace Foo;
                use Exception;
                class Bar {
                    /**
                     * @return array{code: int, exception: Exception}
                     */
                    public function f1(): array {}
                    /**
                     * @return array{exception: Exception}
                     */
                    public function f2(): array {}
                    /**
                     * @return array{exceptions: array<Exception>}
                     */
                    public function f3(): array {}
                }',
            ['import_classes' => false],
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->testFix($expected, $input, $configuration);
    }

    /**
     * @return iterable<array{string, null|string, _AutogeneratedInputConfiguration}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield 'import_classes (true) - try catch without variable' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (Exception) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                try {
                } catch (\Exception) {
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (true) - try catch without variable and comments' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* non-capturing catch */ Exception /* just because! */) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                try {
                } catch (/* non-capturing catch */ \Exception /* just because! */) {
                }
                INPUT,
            ['import_classes' => true],
        ];

        yield 'import_classes (false) - try catch without variable' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (\Exception) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (Exception) {
                }
                INPUT,
            ['import_classes' => false],
        ];

        yield 'import_classes (false) - try catch without variable and comments' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* non-capturing catch */ \Exception /* just because! */) {
                }
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                use Exception;
                try {
                } catch (/* non-capturing catch */ Exception /* just because! */) {
                }
                INPUT,
            ['import_classes' => false],
        ];

        yield [
            <<<'INPUT'
                <?php
                namespace Test;
                echo \FOO, \BAR;

                namespace OtherTest;
                echo \FOO, \BAR;
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield [
            <<<'INPUT'
                <?php
                namespace Test {
                    echo \FOO, \BAR;

                }

                namespace OtherTest {
                    echo \FOO, \BAR;
                }
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield [
            <<<'INPUT'
                <?php
                namespace {
                    echo \FOO, \BAR;

                }

                namespace OtherTest {
                    echo \FOO, \BAR;
                }
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield [
            <<<'INPUT'
                <?php
                namespace Test {
                    echo \FOO, \BAR;

                }

                namespace {
                    echo \FOO, \BAR;
                }
                INPUT,
            null,
            ['import_constants' => true],
        ];

        yield 'attributes' => [
            '<?php
namespace Foo;
use AnAttribute1;
use AnAttribute2;
use AnAttribute3;
class Bar
{
    #[AnAttribute1]
    public function f1() {}
    #[AnAttribute2, AnAttribute3]
    public function f2() {}
}',
            '<?php
namespace Foo;
class Bar
{
    #[\AnAttribute1]
    public function f1() {}
    #[\AnAttribute2, \AnAttribute3]
    public function f2() {}
}',
            [
                'import_classes' => true,
                'import_constants' => true,
                'import_functions' => true,
            ],
        ];
    }

    /**
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, ?string $input = null): void
    {
        $this->fixer->configure([
            'import_constants' => true,
            'import_functions' => true,
        ]);

        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<string, array{string, string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield 'ignore enum methods' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use function foo;
                enum Bar {
                    function foo() {}
                }
                foo();
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                enum Bar {
                    function foo() {}
                }
                \foo();
                INPUT,
        ];

        yield 'ignore enum constants' => [
            <<<'EXPECTED'
                <?php
                namespace Test;
                use const FOO;
                enum Bar {
                    const FOO = 1;
                }
                echo FOO;
                EXPECTED,
            <<<'INPUT'
                <?php
                namespace Test;
                enum Bar {
                    const FOO = 1;
                }
                echo \FOO;
                INPUT,
        ];
    }
}
