<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\LanguageConstruct;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\LanguageConstruct\SingleSpaceAroundConstructFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\LanguageConstruct\SingleSpaceAroundConstructFixer>
 *
 * @author Andreas Möller <am@localheinz.com>
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\LanguageConstruct\SingleSpaceAroundConstructFixer
 */
final class SingleSpaceAroundConstructFixerTest extends AbstractFixerTestCase
{
    /**
     * @dataProvider provideInvalidConfigurationCases
     *
     * @param mixed $construct
     */
    public function testInvalidConfiguration($construct): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);

        $this->fixer->configure([
            'constructs_followed_by_a_single_space' => [
                $construct,
            ],
        ]);
    }

    /**
     * @return iterable<string, array{mixed}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'null' => [null];

        yield 'false' => [false];

        yield 'true' => [true];

        yield 'int' => [0];

        yield 'float' => [3.14];

        yield 'array' => [[]];

        yield 'object' => [new \stdClass()];

        yield 'unknown' => ['foo'];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{string, null|string, _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php abstract class Foo {}; if($a){}',
            '<?php abstract  class Foo {}; if($a){}',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php abstract class Foo {};',
            '<?php abstract

class Foo {};',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php abstract /* foo */class Foo {};',
            '<?php abstract/* foo */class Foo {};',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php abstract /* foo */class Foo {};',
            '<?php abstract  /* foo */class Foo {};',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php

abstract class Foo
{
    abstract function bar();
}',
            '<?php

abstract class Foo
{
    abstract  function bar();
}',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php

abstract class Foo
{
    abstract function bar();
}',
            '<?php

abstract class Foo
{
    abstract

function bar();
}',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php

abstract class Foo
{
    abstract /* foo */function bar();
}',
            '<?php

abstract class Foo
{
    abstract  /* foo */function bar();
}',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php

abstract class Foo
{
    abstract /* foo */function bar();
}',
            '<?php

abstract class Foo
{
    abstract/* foo */function bar();
}',
            ['constructs_followed_by_a_single_space' => ['abstract']],
        ];

        yield [
            '<?php while (true) { break; }',
            null,
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break /* foo */; }',
            '<?php while (true) { break/* foo */; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break /* foo */; }',
            '<?php while (true) { break  /* foo */; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break 1; }',
            '<?php while (true) { break  1; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break 1; }',
            '<?php while (true) { break

1; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break /* foo */1; }',
            '<?php while (true) { break/* foo */1; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php while (true) { break /* foo */1; }',
            '<?php while (true) { break  /* foo */1; }',
            ['constructs_followed_by_a_single_space' => ['break']],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach ($foo as$bar) {}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach ($foo as  $bar) {}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach ($foo  as $bar) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach ($foo as

$bar) {}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach ($foo
as $bar) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach ($foo as /* foo */$bar) {}',
            '<?php foreach ($foo as/* foo */$bar) {}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php foreach ($foo/* foo */ as $bar) {}',
            '<?php foreach ($foo/* foo */as $bar) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach ($foo as /* foo */$bar) {}',
            '<?php foreach ($foo as  /* foo */$bar) {}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php foreach ($foo /* foo */ as $bar) {}',
            '<?php foreach ($foo /* foo */    as $bar) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach (range(1, 12) as $num) {}',
            '<?php foreach (range(1, 12)as $num) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach (range(1, 12) as $num) {}',
            '<?php foreach (range(1, 12)   as $num) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach ([1, 2, 3, 4] as $int) {}',
            '<?php foreach ([1, 2, 3, 4]as $int) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php foreach ([1, 2, 3, 4] as $int) {}',
            '<?php foreach ([1, 2, 3, 4]
                as $int) {}',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php

class Foo
{
    use Bar {
        Bar::baz as bar;
    }
}',
            '<?php

class Foo
{
    use Bar {
        Bar::baz as  bar;
    }
}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php

class Foo
{
    use Bar {
        Bar::baz as bar;
    }
}',
            '<?php

class Foo
{
    use Bar {
        Bar::baz as

bar;
    }
}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php

class Foo
{
    use Bar {
        Bar::baz as /* foo */bar;
    }
}',
            '<?php

class Foo
{
    use Bar {
        Bar::baz as/* foo */bar;
    }
}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php

class Foo
{
    use Bar {
        Bar::baz as /* foo */bar;
    }
}',
            '<?php

class Foo
{
    use Bar {
        Bar::baz as  /* foo */bar;
    }
}',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['as'],
            ],
        ];

        yield [
            '<?php
namespace Foo;

use Bar as Baz;

final class Qux extends Baz {}
',
            '<?php
namespace Foo;

use Bar    as Baz;

final class Qux extends Baz {}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
namespace Foo;

use Bar as Baz;

final class Qux extends Baz {}
',
            '<?php
namespace Foo;

use Bar
    as Baz;

final class Qux extends Baz {}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
namespace Foo;

use Bar /** foo */ as Baz;

final class Qux extends Baz {}
',
            '<?php
namespace Foo;

use Bar /** foo */as Baz;

final class Qux extends Baz {}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
class Foo
{
    use Bar {
        Bar::baz as bar;
    }
}
',
            '<?php
class Foo
{
    use Bar {
        Bar::baz    as bar;
    }
}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
class Foo
{
    use Bar {
        Bar::baz as bar;
    }
}
',
            '<?php
class Foo
{
    use Bar {
        Bar::baz
as bar;
    }
}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
class Foo
{
    use Bar {
        Bar::baz/** foo */ as bar;
    }
}
',
            '<?php
class Foo
{
    use Bar {
        Bar::baz/** foo */as bar;
    }
}
',
            [
                'constructs_preceded_by_a_single_space' => ['as'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php
switch ($i) {
    case $j:
        break;
}',
            '<?php
switch ($i) {
    case$j:
        break;
}',
            ['constructs_followed_by_a_single_space' => ['case']],
        ];

        yield [
            '<?php
switch ($i) {
    case 0:
        break;
}',
            '<?php
switch ($i) {
    case  0:
        break;
}',
            ['constructs_followed_by_a_single_space' => ['case']],
        ];

        yield [
            '<?php
switch ($i) {
    case 0:
        break;
}',
            '<?php
switch ($i) {
    case

0:
        break;
}',
            ['constructs_followed_by_a_single_space' => ['case']],
        ];

        yield [
            '<?php
switch ($i) {
    case /* foo */0:
        break;
}',
            '<?php
switch ($i) {
    case/* foo */0:
        break;
}',
            ['constructs_followed_by_a_single_space' => ['case']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try {} catch(\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['catch']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try {} catch  (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['catch']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try {} catch

(\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['catch']],
        ];

        yield [
            '<?php try {} catch /* foo */(Exception $exception) {}',
            '<?php try {} catch/* foo */(Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['catch']],
        ];

        yield [
            '<?php try {} catch /* foo */(Exception $exception) {}',
            '<?php try {} catch  /* foo */(Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['catch']],
        ];

        yield [
            '<?php class Foo {}',
            '<?php class  Foo {}',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php class Foo {}',
            '<?php class

Foo {}',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php class /* foo */Foo {}',
            '<?php class  /* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php class /* foo */Foo {}',
            '<?php class/* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = stdClass::class;',
            null,
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = new class {};',
            '<?php $foo = new class  {};',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = new class {};',
            '<?php $foo = new class{};',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = new class /* foo */{};',
            '<?php $foo = new class/* foo */{};',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = new class /* foo */{};',
            '<?php $foo = new class  /* foo */{};',
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php $foo = new class(){};',
            null,
            ['constructs_followed_by_a_single_space' => ['class']],
        ];

        yield [
            '<?php return
                    $a ? new class(){ public function foo() { echo 1; }}
                    : 1
                ;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php while (true) { continue; }',
            null,
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php while (true) { continue /* foo */; }',
            '<?php while (true) { continue/* foo */; }',
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php while (true) { continue /* foo */; }',
            '<?php while (true) { continue  /* foo */; }',
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php while (true) { continue 1; }',
            '<?php while (true) { continue  1; }',
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php while (true) { continue 1; }',
            '<?php while (true) { continue

1; }',
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php while (true) { continue /* foo*/ 1; }',
            '<?php while (true) { continue  /* foo*/ 1; }',
            ['constructs_followed_by_a_single_space' => ['continue']],
        ];

        yield [
            '<?php class Foo { const FOO = 9000; }',
            '<?php class Foo { const  FOO = 9000; }',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php class Foo { const FOO = 9000; }',
            '<?php class Foo { const

FOO = 9000; }',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php class Foo { const /* foo */FOO = 9000; }',
            '<?php class Foo { const/* foo */FOO = 9000; }',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php class Foo { const /* foo */FOO = 9000; }',
            '<?php class Foo { const  /* foo */FOO = 9000; }',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php class Foo {
                const FOO = [
                    1
                ];
                const BAR = [
                    2,
                ];
            }',
            '<?php class Foo {
                const    FOO = [
                    1
                ];
                const    BAR = [
                    2,
                ];
            }',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield ['<?php class Foo {
    const
        FOO = 9000,
        BAR = 10000;
}',
            null,
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php
const
    A = 3,
    B = 3
?>',
            null,
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php
const A = 3 ?>

<?php
[ ,
,
,$z
] = foo()  ;',
            '<?php
const     A = 3 ?>

<?php
[ ,
,
,$z
] = foo()  ;',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php
    const A
    =
    1;
',
            '<?php
    const
    A
    =
    1;
',
            ['constructs_followed_by_a_single_space' => ['const']],
        ];

        yield [
            '<?php use const FOO\BAR;',
            '<?php use const  FOO\BAR;',
            ['constructs_followed_by_a_single_space' => ['const_import']],
        ];

        yield [
            '<?php use const FOO\BAR;',
            '<?php use const

FOO\BAR;',
            ['constructs_followed_by_a_single_space' => ['const_import']],
        ];

        yield [
            '<?php use const /* foo */FOO\BAR;',
            '<?php use const/* foo */FOO\BAR;',
            ['constructs_followed_by_a_single_space' => ['const_import']],
        ];

        yield [
            '<?php use const /* foo */FOO\BAR;',
            '<?php use const  /* foo */FOO\BAR;',
            ['constructs_followed_by_a_single_space' => ['const_import']],
        ];

        yield [
            '<?php clone $foo;',
            '<?php clone$foo;',
            ['constructs_followed_by_a_single_space' => ['clone']],
        ];

        yield [
            '<?php clone $foo;',
            '<?php clone  $foo;',
            ['constructs_followed_by_a_single_space' => ['clone']],
        ];

        yield [
            '<?php clone $foo;',
            '<?php clone

$foo;',
            ['constructs_followed_by_a_single_space' => ['clone']],
        ];

        yield [
            '<?php clone /* foo */$foo;',
            '<?php clone/* foo */$foo;',
            ['constructs_followed_by_a_single_space' => ['clone']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do{} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php DO {} while (true);',
            '<?php DO{} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do  {} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do

{} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php do /* foo*/{} while (true);',
            '<?php do/* foo*/{} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php do /* foo*/{} while (true);',
            '<?php do  /* foo*/{} while (true);',
            ['constructs_followed_by_a_single_space' => ['do']],
        ];

        yield [
            '<?php echo $foo;',
            '<?php echo$foo;',
            ['constructs_followed_by_a_single_space' => ['echo']],
        ];

        yield [
            '<?php echo 9000;',
            '<?php echo  9000;',
            ['constructs_followed_by_a_single_space' => ['echo']],
        ];

        yield [
            '<?php echo 9000;',
            '<?php echo

9000;',
            ['constructs_followed_by_a_single_space' => ['echo']],
        ];

        yield [
            '<?php ECHO /* foo */9000;',
            '<?php ECHO/* foo */9000;',
            ['constructs_followed_by_a_single_space' => ['echo']],
        ];

        yield [
            '<?php if (true) {} else {}',
            '<?php if (true) {} else{}',
            ['constructs_followed_by_a_single_space' => ['else']],
        ];

        yield [
            '<?php if (true) {} else {}',
            '<?php if (true) {} else  {}',
            ['constructs_followed_by_a_single_space' => ['else']],
        ];

        yield [
            '<?php if (true){} else {}',
            '<?php if (true){} else

{}',
            ['constructs_followed_by_a_single_space' => ['else']],
        ];

        yield [
            '<?php if (true) {} else /* foo */{}',
            '<?php if (true) {} else/* foo */{}',
            ['constructs_followed_by_a_single_space' => ['else']],
        ];

        yield [
            '<?php if ($a222) {} else {}',
            '<?php if ($a222) {}else{}',
            ['constructs_preceded_by_a_single_space' => ['else']],
        ];

        yield [
            "<?php if (\$b333) {}//foo X\nelse {}",
            null,
            ['constructs_preceded_by_a_single_space' => ['else']],
        ];

        yield [
            '<?php if (true) {}elseif (false) {}',
            '<?php if (true) {}elseif(false) {}',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true) {} elseif (false) {}',
            '<?php if (true) {} elseif  (false) {}',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true) {} elseif (false) {}',
            '<?php if (true) {} elseif

(false) {}',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true) {} elseif /* foo */(false) {}',
            '<?php if (true) {} elseif/* foo */(false) {}',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true): elseif (false): endif;',
            '<?php if (true): elseif(false): endif;',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true): elseif (false): endif;',
            '<?php if (true): elseif  (false): endif;',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true): elseif (false): endif;',
            '<?php if (true): elseif

(false): endif;',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if (true): elseif /* foo */(false): endif;',
            '<?php if (true): elseif/* foo */(false): endif;',
            ['constructs_followed_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if ($y1x) {} elseif (false) {}',
            '<?php if ($y1x) {}elseif(false) {}',
            ['constructs_preceded_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php if ($y2y) {} elseif (false) {}',
            '<?php if ($y2y) {}    elseif   (false) {}',
            ['constructs_preceded_by_a_single_space' => ['elseif']],
        ];

        yield [
            "<?php if (\$y3t) {}//foo Y\nelseif (false) {}",
            null,
            ['constructs_preceded_by_a_single_space' => ['elseif']],
        ];

        yield [
            '<?php class Foo extends \InvalidArgumentException {}',
            '<?php class Foo extends  \InvalidArgumentException {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php class Foo extends \InvalidArgumentException {}',
            '<?php class Foo extends

\InvalidArgumentException {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php class Foo extends /* foo */\InvalidArgumentException {}',
            '<?php class Foo extends/* foo */\InvalidArgumentException {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php class Foo extends /* foo */\InvalidArgumentException {}',
            '<?php class Foo extends  /* foo */\InvalidArgumentException {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends Bar1 {}',
            '<?php interface Foo extends  Bar1 {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends Bar2 {}',
            '<?php interface Foo extends

Bar2 {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends /* foo */Bar3 {}',
            '<?php interface Foo extends/* foo */Bar3 {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends /* foo */Bar4 {}',
            '<?php interface Foo extends  /* foo */Bar4 {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends Bar5, Baz, Qux {}',
            '<?php interface Foo extends  Bar5, Baz, Qux {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends Bar6, Baz, Qux {}',
            '<?php interface Foo extends

Bar6, Baz, Qux {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends /* foo */Bar7, Baz, Qux {}',
            '<?php interface Foo extends/* foo */Bar7, Baz, Qux {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends /* foo */Bar8, Baz, Qux {}',
            '<?php interface Foo extends  /* foo */Bar8, Baz, Qux {}',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php interface Foo extends
    Bar9,
    Baz,
    Qux
{}',
            null,
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php $foo = new class extends \InvalidArgumentException {};',
            '<?php $foo = new class extends  \InvalidArgumentException {};',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php $foo = new class extends \InvalidArgumentException {};',
            '<?php $foo = new class extends

\InvalidArgumentException {};',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php $foo = new class extends /* foo */\InvalidArgumentException {};',
            '<?php $foo = new class extends/* foo */\InvalidArgumentException {};',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php $foo = new class extends /* foo */\InvalidArgumentException {};',
            '<?php $foo = new class extends  /* foo */\InvalidArgumentException {};',
            ['constructs_followed_by_a_single_space' => ['extends']],
        ];

        yield [
            '<?php final class Foo {}',
            '<?php final  class Foo {}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php final class Foo {}',
            '<?php final

class Foo {}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php final /* foo */class Foo {}',
            '<?php final/* foo */class Foo {}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php final /* foo */class Foo {}',
            '<?php final  /* foo */class Foo {}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php

class Foo
{
    final function bar() {}
}',
            '<?php

class Foo
{
    final  function bar() {}
}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php

class Foo
{
    final function bar() {}
}',
            '<?php

class Foo
{
    final

function bar() {}
}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php

class Foo
{
    final /* foo */function bar() {}
}',
            '<?php

class Foo
{
    final/* foo */function bar() {}
}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php

class Foo
{
    final /* foo */function bar() {}
}',
            '<?php

class Foo
{
    final  /* foo */function bar() {}
}',
            ['constructs_followed_by_a_single_space' => ['final']],
        ];

        yield [
            '<?php try {} finally {}',
            '<?php try {} finally{}',
            ['constructs_followed_by_a_single_space' => ['finally']],
        ];

        yield [
            '<?php try {} finally {}',
            '<?php try {} finally  {}',
            ['constructs_followed_by_a_single_space' => ['finally']],
        ];

        yield [
            '<?php try {} finally {}',
            '<?php try {} finally

{}',
            ['constructs_followed_by_a_single_space' => ['finally']],
        ];

        yield [
            '<?php try {} finally /* foo */{}',
            '<?php try {} finally/* foo */{}',
            ['constructs_followed_by_a_single_space' => ['finally']],
        ];

        yield [
            '<?php try {} finally /* foo */{}',
            '<?php try {} finally  /* foo */{}',
            ['constructs_followed_by_a_single_space' => ['finally']],
        ];

        yield [
            '<?php for ($i = 0; $i < 3; ++$i) {}',
            '<?php for($i = 0; $i < 3; ++$i) {}',
            ['constructs_followed_by_a_single_space' => ['for']],
        ];

        yield [
            '<?php for ($i = 0; $i < 3; ++$i) {}',
            '<?php for  ($i = 0; $i < 3; ++$i) {}',
            ['constructs_followed_by_a_single_space' => ['for']],
        ];

        yield [
            '<?php for ($i = 0; $i < 3; ++$i) {}',
            '<?php for

($i = 0; $i < 3; ++$i) {}',
            ['constructs_followed_by_a_single_space' => ['for']],
        ];

        yield [
            '<?php for /* foo */($i = 0; $i < 3; ++$i) {}',
            '<?php for/* foo */($i = 0; $i < 3; ++$i) {}',
            ['constructs_followed_by_a_single_space' => ['for']],
        ];

        yield [
            '<?php for /* foo */($i = 0; $i < 3; ++$i) {}',
            '<?php for  /* foo */($i = 0; $i < 3; ++$i) {}',
            ['constructs_followed_by_a_single_space' => ['for']],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach($foo as $bar) {}',
            ['constructs_followed_by_a_single_space' => ['foreach']],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach  ($foo as $bar) {}',
            ['constructs_followed_by_a_single_space' => ['foreach']],
        ];

        yield [
            '<?php foreach ($foo as $bar) {}',
            '<?php foreach

($foo as $bar) {}',
            ['constructs_followed_by_a_single_space' => ['foreach']],
        ];

        yield [
            '<?php foreach /* foo */($foo as $bar) {}',
            '<?php foreach/* foo */($foo as $bar) {}',
            ['constructs_followed_by_a_single_space' => ['foreach']],
        ];

        yield [
            '<?php foreach /* foo */($foo as $bar) {}',
            '<?php foreach  /* foo */($foo as $bar) {}',
            ['constructs_followed_by_a_single_space' => ['foreach']],
        ];

        yield [
            '<?php function foo() {}',
            '<?php function  foo() {}',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php function foo() {}',
            '<?php function

foo() {}',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php function /* foo */foo() {}',
            '<?php function/* foo */foo() {}',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php function /* foo */foo() {}',
            '<?php function  /* foo */foo() {}',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php
class Foo
{
    function bar() {}
}
',
            '<?php
class Foo
{
    function  bar() {}
}
',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php
class Foo
{
    function bar() {}
}
',
            '<?php
class Foo
{
    function

bar() {}
}
',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php
class Foo
{
    function /* foo */bar() {}
}
',
            '<?php
class Foo
{
    function/* foo */bar() {}
}
',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php
class Foo
{
    function /* foo */bar() {}
}
',
            '<?php
class Foo
{
    function  /* foo */bar() {}
}
',
            ['constructs_followed_by_a_single_space' => ['function']],
        ];

        yield [
            '<?php use function Foo\bar;',
            '<?php use function  Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['function_import']],
        ];

        yield [
            '<?php use function Foo\bar;',
            '<?php use function

Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['function_import']],
        ];

        yield [
            '<?php use function /* foo */Foo\bar;',
            '<?php use function/* foo */Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['function_import']],
        ];

        yield [
            '<?php use function /* foo */Foo\bar;',
            '<?php use function  /* foo */Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['function_import']],
        ];

        yield [
            '<?php function foo() { global $bar; }',
            '<?php function foo() { global$bar; }',
            ['constructs_followed_by_a_single_space' => ['global']],
        ];

        yield [
            '<?php function foo() { global $bar; }',
            '<?php function foo() { global  $bar; }',
            ['constructs_followed_by_a_single_space' => ['global']],
        ];

        yield [
            '<?php function foo() { global $bar; }',
            '<?php function foo() { global

$bar; }',
            ['constructs_followed_by_a_single_space' => ['global']],
        ];

        yield [
            '<?php function foo() { global /* foo */$bar; }',
            '<?php function foo() { global/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['global']],
        ];

        yield [
            '<?php function foo() { global /* foo */$bar; }',
            '<?php function foo() { global  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['global']],
        ];

        yield [
            '<?php goto foo; foo: echo "Bar";',
            '<?php goto  foo; foo: echo "Bar";',
            ['constructs_followed_by_a_single_space' => ['goto']],
        ];

        yield [
            '<?php goto foo; foo: echo "Bar";',
            '<?php goto

foo; foo: echo "Bar";',
            ['constructs_followed_by_a_single_space' => ['goto']],
        ];

        yield [
            '<?php goto /* foo */foo; foo: echo "Bar";',
            '<?php goto/* foo */foo; foo: echo "Bar";',
            ['constructs_followed_by_a_single_space' => ['goto']],
        ];

        yield [
            '<?php if ($foo === $bar) {}',
            '<?php if($foo === $bar) {}',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if ($foo === $bar) {}',
            '<?php if  ($foo === $bar) {}',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if ($foo === $bar) {}',
            '<?php if

($foo === $bar) {}',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if /* foo */($foo === $bar) {}',
            '<?php if/* foo */($foo === $bar) {}',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if ($foo === $bar): endif;',
            '<?php if($foo === $bar): endif;',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if ($foo === $bar): endif;',
            '<?php if  ($foo === $bar): endif;',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if ($foo === $bar): endif;',
            '<?php if

($foo === $bar): endif;',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if /* foo */($foo === $bar): endif;',
            '<?php if/* foo */($foo === $bar): endif;',
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php if (true): else: endif;',
            null,
            ['constructs_followed_by_a_single_space' => ['if']],
        ];

        yield [
            '<?php class Foo implements \Countable {}',
            '<?php class Foo implements  \Countable {}',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php class Foo implements \Countable {}',
            '<?php class Foo implements

\Countable {}',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php class Foo implements /* foo */\Countable {}',
            '<?php class Foo implements/* foo */\Countable {}',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php class Foo implements /* foo */\Countable {}',
            '<?php class Foo implements  /* foo */\Countable {}',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php class Foo implements
                    \Countable,
                    Bar,
                    Baz
                {}',
            null,
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php $foo = new class implements \Countable {};',
            '<?php $foo = new class implements  \Countable {};',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php $foo = new class implements \Countable {};',
            '<?php $foo = new class implements

\Countable {};',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php $foo = new class implements /* foo */\Countable {};',
            '<?php $foo = new class implements/* foo */\Countable {};',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php $foo = new class implements /* foo */\Countable {};',
            '<?php $foo = new class implements  /* foo */\Countable {};',
            ['constructs_followed_by_a_single_space' => ['implements']],
        ];

        yield [
            '<?php include "vendor/autoload.php";',
            '<?php include"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include']],
        ];

        yield [
            '<?php include "vendor/autoload.php";',
            '<?php include  "vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include']],
        ];

        yield [
            '<?php include "vendor/autoload.php";',
            '<?php include

"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include']],
        ];

        yield [
            '<?php include /* foo */"vendor/autoload.php";',
            '<?php include/* foo */"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include']],
        ];

        yield [
            '<?php include_once "vendor/autoload.php";',
            '<?php include_once"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include_once']],
        ];

        yield [
            '<?php include_once "vendor/autoload.php";',
            '<?php include_once  "vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include_once']],
        ];

        yield [
            '<?php include_once "vendor/autoload.php";',
            '<?php include_once

"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include_once']],
        ];

        yield [
            '<?php include_once /* foo */"vendor/autoload.php";',
            '<?php include_once/* foo */"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['include_once']],
        ];

        yield [
            '<?php $foo instanceof \stdClass;',
            '<?php $foo instanceof  \stdClass;',
            ['constructs_followed_by_a_single_space' => ['instanceof']],
        ];

        yield [
            '<?php $foo instanceof \stdClass;',
            '<?php $foo instanceof

\stdClass;',
            ['constructs_followed_by_a_single_space' => ['instanceof']],
        ];

        yield [
            '<?php $foo instanceof /* foo */\stdClass;',
            '<?php $foo instanceof/* foo */\stdClass;',
            ['constructs_followed_by_a_single_space' => ['instanceof']],
        ];

        yield [
            '<?php $foo instanceof /* foo */\stdClass;',
            '<?php $foo instanceof  /* foo */\stdClass;',
            ['constructs_followed_by_a_single_space' => ['instanceof']],
        ];

        yield [
            '<?php $foo instanceof $bar;',
            '<?php $foo instanceof$bar;',
            ['constructs_followed_by_a_single_space' => ['instanceof']],
        ];

        yield [
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof A;
        A::bigTalk insteadof B;
    }
}',
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof  A;
        A::bigTalk insteadof B;
    }
}',
            ['constructs_followed_by_a_single_space' => ['insteadof']],
        ];

        yield [
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof A;
        A::bigTalk insteadof B;
    }
}',
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof

A;
        A::bigTalk insteadof B;
    }
}',
            ['constructs_followed_by_a_single_space' => ['insteadof']],
        ];

        yield [
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof /* foo */A;
        A::bigTalk insteadof B;
    }
}',
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof/* foo */A;
        A::bigTalk insteadof B;
    }
}',
            ['constructs_followed_by_a_single_space' => ['insteadof']],
        ];

        yield [
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof /* foo */A;
        A::bigTalk insteadof B;
    }
}',
            '<?php

class Talker {
    use A, B {
        B::smallTalk insteadof  /* foo */A;
        A::bigTalk insteadof B;
    }
}',
            ['constructs_followed_by_a_single_space' => ['insteadof']],
        ];

        yield [
            '<?php interface Foo {}',
            '<?php interface  Foo {}',
            ['constructs_followed_by_a_single_space' => ['interface']],
        ];

        yield [
            '<?php interface Foo {}',
            '<?php interface

Foo {}',
            ['constructs_followed_by_a_single_space' => ['interface']],
        ];

        yield [
            '<?php interface /* foo */Foo {}',
            '<?php interface  /* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['interface']],
        ];

        yield [
            '<?php interface /* foo */Foo {}',
            '<?php interface/* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['interface']],
        ];

        yield [
            '<?php new $foo();',
            '<?php new$foo();',
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield [
            '<?php new Bar();',
            '<?php new  Bar();',
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield [
            '<?php new Bar();',
            '<?php new

Bar();',
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield [
            '<?php new /* foo */Bar();',
            '<?php new/* foo */Bar();',
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield 'attribute on separate line' => [
            <<<'EOF'
                <?php
                $a = new
                #[FOO]
                class() {};
                EOF,
            null,
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield 'phpdoc on separate line' => [
            <<<'EOF'
                <?php
                $a = new
                /**
                 @property string $x
                 */
                class() {};
                EOF,
            null,
            ['constructs_followed_by_a_single_space' => ['new']],
        ];

        yield [
            '<?= $foo ?>',
            '<?=$foo ?>',
            ['constructs_followed_by_a_single_space' => ['open_tag_with_echo']],
        ];

        yield [
            '<?= $foo ?>',
            '<?=  $foo ?>',
            ['constructs_followed_by_a_single_space' => ['open_tag_with_echo']],
        ];

        yield [
            '<?= $foo ?>',
            '<?=

$foo ?>',
            ['constructs_followed_by_a_single_space' => ['open_tag_with_echo']],
        ];

        yield [
            '<?= /* foo */$foo ?>',
            '<?=/* foo */$foo ?>',
            ['constructs_followed_by_a_single_space' => ['open_tag_with_echo']],
        ];

        yield [
            '<?= /* foo */$foo ?>',
            '<?=  /* foo */$foo ?>',
            ['constructs_followed_by_a_single_space' => ['open_tag_with_echo']],
        ];

        yield [
            '<?php print $foo;',
            '<?php print$foo;',
            ['constructs_followed_by_a_single_space' => ['print']],
        ];

        yield [
            '<?php print 9000;',
            '<?php print  9000;',
            ['constructs_followed_by_a_single_space' => ['print']],
        ];

        yield [
            '<?php print 9000;',
            '<?php print

9000;',
            ['constructs_followed_by_a_single_space' => ['print']],
        ];

        yield [
            '<?php print /* foo */9000;',
            '<?php print/* foo */9000;',
            ['constructs_followed_by_a_single_space' => ['print']],
        ];

        yield [
            '<?php class Foo { private $bar; }',
            '<?php class Foo { private$bar; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private $bar; }',
            '<?php class Foo { private  $bar; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private $bar; }',
            '<?php class Foo { private

$bar; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */$bar; }',
            '<?php class Foo { private/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */$bar; }',
            '<?php class Foo { private  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private function bar() {} }',
            '<?php class Foo { private  function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private function bar() {} }',
            '<?php class Foo { private

function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */function bar() {} }',
            '<?php class Foo { private/* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */function bar() {} }',
            '<?php class Foo { private  /* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private CONST BAR = 9000; }',
            '<?php class Foo { private  CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private CONST BAR = 9000; }',
            '<?php class Foo { private

CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */CONST BAR = 9000; }',
            '<?php class Foo { private/* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { private /* foo */CONST BAR = 9000; }',
            '<?php class Foo { private  /* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['private']],
        ];

        yield [
            '<?php class Foo { protected $bar; }',
            '<?php class Foo { protected$bar; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected $bar; }',
            '<?php class Foo { protected  $bar; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected $bar; }',
            '<?php class Foo { protected

$bar; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */$bar; }',
            '<?php class Foo { protected/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */$bar; }',
            '<?php class Foo { protected  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected function bar() {} }',
            '<?php class Foo { protected  function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected function bar() {} }',
            '<?php class Foo { protected

function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */function bar() {} }',
            '<?php class Foo { protected/* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */function bar() {} }',
            '<?php class Foo { protected  /* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected CONST BAR = 9000; }',
            '<?php class Foo { protected  CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected CONST BAR = 9000; }',
            '<?php class Foo { protected

CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */CONST BAR = 9000; }',
            '<?php class Foo { protected/* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { protected /* foo */CONST BAR = 9000; }',
            '<?php class Foo { protected  /* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['protected']],
        ];

        yield [
            '<?php class Foo { public $bar; }',
            '<?php class Foo { public$bar; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { Public $bar; }',
            '<?php class Foo { Public  $bar; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public $bar; }',
            '<?php class Foo { public

$bar; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */$bar; }',
            '<?php class Foo { public/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */$bar; }',
            '<?php class Foo { public  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public function bar() {} }',
            '<?php class Foo { public  function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public function bar() {} }',
            '<?php class Foo { public

function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */function bar() {} }',
            '<?php class Foo { public/* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */function bar() {} }',
            '<?php class Foo { public  /* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public CONST BAR = 9000; }',
            '<?php class Foo { public  CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public CONST BAR = 9000; }',
            '<?php class Foo { public

CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */CONST BAR = 9000; }',
            '<?php class Foo { public/* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php class Foo { public /* foo */CONST BAR = 9000; }',
            '<?php class Foo { public  /* foo */CONST BAR = 9000; }',
            ['constructs_followed_by_a_single_space' => ['public']],
        ];

        yield [
            '<?php require "vendor/autoload.php";',
            '<?php require"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require']],
        ];

        yield [
            '<?php require "vendor/autoload.php";',
            '<?php require  "vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require']],
        ];

        yield [
            '<?php require "vendor/autoload.php";',
            '<?php require

"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require']],
        ];

        yield [
            '<?php require /* foo */"vendor/autoload.php";',
            '<?php require/* foo */"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require']],
        ];

        yield [
            '<?php require_once "vendor/autoload.php";',
            '<?php require_once"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require_once']],
        ];

        yield [
            '<?php require_once "vendor/autoload.php";',
            '<?php require_once  "vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require_once']],
        ];

        yield [
            '<?php require_once "vendor/autoload.php";',
            '<?php require_once

"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require_once']],
        ];

        yield [
            '<?php require_once /* foo */"vendor/autoload.php";',
            '<?php require_once/* foo */"vendor/autoload.php";',
            ['constructs_followed_by_a_single_space' => ['require_once']],
        ];

        yield [
            '<?php return;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return /* foo */;',
            '<?php return/* foo */;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return /* foo */;',
            '<?php return  /* foo */;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return $foo;',
            '<?php return$foo;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return 9000;',
            '<?php return  9000;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return 9000;',
            '<?php return

9000;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return /* */ 9000 + 1 /* foo */       ?>',
            '<?php return





/* */ 9000 + 1 /* foo */       ?>',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return /* foo */9000;',
            '<?php return/* foo */9000;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php return $foo && $bar || $baz;',
            '<?php return

$foo && $bar || $baz;',
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo
    && $bar
    || $baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo &&
    $bar ||
    $baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo
    + $bar
    - $baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo +
    $bar -
    $baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo ?
    $bar :
    $baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo
    ? $bar
    : baz;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo ?:
    $bar;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo
    ?: $bar;',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php

return
    $foo
    ?: $bar?>',
            null,
            ['constructs_followed_by_a_single_space' => ['return']],
        ];

        yield [
            '<?php function foo() { static $bar; }',
            '<?php function foo() { static$bar; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php function foo() { static $bar; }',
            '<?php function foo() { static  $bar; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php function foo() { static $bar; }',
            '<?php function foo() { static

$bar; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php function foo() { static /* foo */$bar; }',
            '<?php function foo() { static/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php function foo() { static /* foo */$bar; }',
            '<?php function foo() { static  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { static function bar() {} }',
            '<?php class Foo { static  function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { static function bar() {} }',
            '<?php class Foo { static

function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { static /* foo */function bar() {} }',
            '<?php class Foo { static/* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { static /* foo */function bar() {} }',
            '<?php class Foo { static  /* foo */function bar() {} }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { public static ?int $x; }',
            '<?php class Foo { public static?int $x; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { public static ?int $x; }',
            '<?php class Foo { public static   ?int $x; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { public static int $x; }',
            '<?php class Foo { public static   int $x; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { public static \Closure $a; }',
            '<?php class Foo { public static    \Closure $a; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { public static array $c; }',
            '<?php class Foo { public static
array $c; }',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php $a = static fn(): bool => true;',
            '<?php $a = static    fn(): bool => true;',
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { function bar() { return new static(); } }',
            null,
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php class Foo { function bar() { return static::class; } }',
            null,
            ['constructs_followed_by_a_single_space' => ['static']],
        ];

        yield [
            '<?php throw $foo;',
            '<?php throw$foo;',
            ['constructs_followed_by_a_single_space' => ['throw']],
        ];

        yield [
            '<?php throw new Exception();',
            '<?php throw  new Exception();',
            ['constructs_followed_by_a_single_space' => ['throw']],
        ];

        yield [
            '<?php throw new Exception();',
            '<?php throw

new Exception();',
            ['constructs_followed_by_a_single_space' => ['throw']],
        ];

        yield [
            '<?php throw /* foo */new Exception();',
            '<?php throw/* foo */new Exception();',
            ['constructs_followed_by_a_single_space' => ['throw']],
        ];

        yield [
            '<?php trait Foo {}',
            '<?php trait  Foo {}',
            ['constructs_followed_by_a_single_space' => ['trait']],
        ];

        yield [
            '<?php trait Foo {}',
            '<?php trait

Foo {}',
            ['constructs_followed_by_a_single_space' => ['trait']],
        ];

        yield [
            '<?php trait /* foo */Foo {}',
            '<?php trait  /* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['trait']],
        ];

        yield [
            '<?php trait /* foo */Foo {}',
            '<?php trait/* foo */Foo {}',
            ['constructs_followed_by_a_single_space' => ['trait']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try{} catch (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['try']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try  {} catch (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['try']],
        ];

        yield [
            '<?php try {} catch (\Exception $exception) {}',
            '<?php try

{} catch (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['try']],
        ];

        yield [
            '<?php try /* foo */{} catch (\Exception $exception) {}',
            '<?php try/* foo */{} catch (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['try']],
        ];

        yield [
            '<?php try /* foo */{} catch (\Exception $exception) {}',
            '<?php try  /* foo */{} catch (\Exception $exception) {}',
            ['constructs_followed_by_a_single_space' => ['try']],
        ];

        yield [
            '<?php use Foo\Bar;',
            '<?php use  Foo\Bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use Foo\Bar;',
            '<?php use

Foo\Bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use /* foo */Foo\Bar;',
            '<?php use/* foo */Foo\Bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use /* foo */Foo\Bar;',
            '<?php use  /* foo */Foo\Bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use const Foo\BAR;',
            '<?php use  const Foo\BAR;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use const Foo\BAR;',
            '<?php use

const Foo\BAR;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use /* foo */const Foo\BAR;',
            '<?php use/* foo */const Foo\BAR;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use function Foo\bar;',
            '<?php use  function Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use function Foo\bar;',
            '<?php use

function Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php use /* foo */function Foo\bar;',
            '<?php use/* foo */function Foo\bar;',
            ['constructs_followed_by_a_single_space' => ['use']],
        ];

        yield [
            '<?php $foo = function () use($bar) {};',
            '<?php $foo = function ()use($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => ['use_lambda'],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php $foo = function ()use ($bar) {};',
            '<?php $foo = function ()use($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php $foo = function () use ($bar) {};',
            '<?php $foo = function ()use($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => ['use_lambda'],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php $foo = function () use ($bar) {};',
            '<?php $foo = function () use  ($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php $foo = function () use ($bar) {};',
            '<?php $foo = function () use

($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php $foo = function () use /* foo */($bar) {};',
            '<?php $foo = function () use/* foo */($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php $foo = function () use /* foo */($bar) {};',
            '<?php $foo = function () use  /* foo */($bar) {};',
            [
                'constructs_preceded_by_a_single_space' => [],
                'constructs_followed_by_a_single_space' => ['use_lambda'],
            ],
        ];

        yield [
            '<?php class Foo { use Bar; }',
            '<?php class Foo { use  Bar; }',
            ['constructs_followed_by_a_single_space' => ['use_trait']],
        ];

        yield [
            '<?php class Foo { use Bar; }',
            '<?php class Foo { use

Bar; }',
            ['constructs_followed_by_a_single_space' => ['use_trait']],
        ];

        yield [
            '<?php class Foo { use /* foo */Bar; }',
            '<?php class Foo { use/* foo */Bar; }',
            ['constructs_followed_by_a_single_space' => ['use_trait']],
        ];

        yield [
            '<?php class Foo { use /* foo */Bar; }',
            '<?php class Foo { use  /* foo */Bar; }',
            ['constructs_followed_by_a_single_space' => ['use_trait']],
        ];

        yield [
            '<?php class Foo { var $bar; }',
            '<?php class Foo { var$bar; }',
            ['constructs_followed_by_a_single_space' => ['var']],
        ];

        yield [
            '<?php class Foo { var $bar; }',
            '<?php class Foo { var  $bar; }',
            ['constructs_followed_by_a_single_space' => ['var']],
        ];

        yield [
            '<?php class Foo { var $bar; }',
            '<?php class Foo { var

$bar; }',
            ['constructs_followed_by_a_single_space' => ['var']],
        ];

        yield [
            '<?php class Foo { var /* foo */$bar; }',
            '<?php class Foo { var/* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['var']],
        ];

        yield [
            '<?php class Foo { var /* foo */$bar; }',
            '<?php class Foo { var  /* foo */$bar; }',
            ['constructs_followed_by_a_single_space' => ['var']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do {} while(true);',
            ['constructs_followed_by_a_single_space' => ['while']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do {} while  (true);',
            ['constructs_followed_by_a_single_space' => ['while']],
        ];

        yield [
            '<?php do {} while (true);',
            '<?php do {} while

(true);',
            ['constructs_followed_by_a_single_space' => ['while']],
        ];

        yield [
            '<?php do {} while /* foo */(true);',
            '<?php do {} while/* foo */(true);',
            ['constructs_followed_by_a_single_space' => ['while']],
        ];

        yield [
            '<?php do {} while /* foo */(true);',
            '<?php do {} while  /* foo */(true);',
            ['constructs_followed_by_a_single_space' => ['while']],
        ];

        yield [
            '<?php function foo() { yield $foo; }',
            '<?php function foo() { yield$foo; }',
            ['constructs_followed_by_a_single_space' => ['yield']],
        ];

        yield [
            '<?php function foo() { yield "Foo"; }',
            '<?php function foo() { yield  "Foo"; }',
            ['constructs_followed_by_a_single_space' => ['yield']],
        ];

        yield [
            '<?php function foo() { yield "Foo"; }',
            '<?php function foo() { yield

"Foo"; }',
            ['constructs_followed_by_a_single_space' => ['yield']],
        ];

        yield [
            '<?php function foo() { yield /* foo */"Foo"; }',
            '<?php function foo() { yield/* foo */"Foo"; }',
            ['constructs_followed_by_a_single_space' => ['yield']],
        ];

        yield [
            '<?php function foo() { yield from $foo; }',
            '<?php function foo() { yield from$foo; }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from baz(); }',
            '<?php function foo() { yield from  baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yIeLd fRoM baz(); }',
            '<?php function foo() { yIeLd fRoM  baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from baz(); }',
            '<?php function foo() { yield from

baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from /* foo */baz(); }',
            '<?php function foo() { yield from/* foo */baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from /* foo */baz(); }',
            '<?php function foo() { yield from  /* foo */baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from /* foo */baz(); }',
            '<?php function foo() { yield from

/* foo */baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield  from baz(); }',
            '<?php function foo() { yield  from  baz(); }',
            [
                'constructs_contain_a_single_space' => [],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from  baz(); }',
            '<?php function foo() { yield  from  baz(); }',
            [
                'constructs_contain_a_single_space' => [
                    'yield_from',
                ],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php function foo() { yield from baz(); }',
            '<?php function foo() { yield  from  baz(); }',
            [
                'constructs_contain_a_single_space' => [
                    'yield_from',
                ],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yIeLd fRoM baz(); }',
            '<?php function foo() { yIeLd  fRoM  baz(); }',
            [
                'constructs_contain_a_single_space' => [
                    'yield_from',
                ],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php function foo() { yield from baz(); }',
            '<?php function foo() { yield

from baz(); }',
            [
                'constructs_contain_a_single_space' => [
                    'yield_from',
                ],
                'constructs_followed_by_a_single_space' => [],
            ],
        ];

        yield [
            '<?php function foo() { yield from baz(); }',
            '<?php function foo() { yield

from

baz(); }',
            [
                'constructs_contain_a_single_space' => [
                    'yield_from',
                ],
                'constructs_followed_by_a_single_space' => [
                    'yield_from',
                ],
            ],
        ];

        yield [
            '<?php echo 1;',
            '<?php    echo 1;',
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            "<?php\necho 1;",
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            "<?php\n   echo 1;",
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            '<?php ',
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            "<?php\n",
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            "<?php \necho 1;",
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            "<?php    \n\necho 1;",
            null,
            ['constructs_followed_by_a_single_space' => ['php_open']],
        ];

        yield [
            '<?php
$a /* 1 */ = /**/ 1;
$a /** 1 */ = /** 2 */ 1;

$a = 3; # 3
$a = 4; /** 4 */
echo 1;
',
            '<?php
$a /* 1 */= /**/1;
$a /** 1 */= /** 2 */1;

$a = 3; # 3
$a = 4; /** 4 */
echo 1;
',
            ['constructs_followed_by_a_single_space' => ['comment', 'php_doc']],
        ];

        yield 'exceptions' => [
            '<?php
new Dummy(/* a */);
new Dummy(/** b */);
foo(/* c */);
foo($a /* d */, $b);
$arr = [/* empty */];
',
            null,
            ['constructs_followed_by_a_single_space' => ['comment', 'php_doc']],
        ];

        yield 'before_destructuring_square_brace_close' => [
            '<?php
foreach ($fields as [$field/** @var string*/]) {
}
',
            null,
            ['constructs_followed_by_a_single_space' => ['comment', 'php_doc']],
        ];

        yield 'simple' => [
            '<?php
namespace Foo;
namespace Bar;',
            '<?php
namespace    Foo;
namespace    Bar;',
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield 'simple with newlines' => [
            '<?php
namespace Foo;
namespace Bar;',
            '<?php
namespace
Foo;
namespace
Bar;',
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield 'braces' => [
            '<?php
namespace Foo {}
namespace Bar {}',
            '<?php
namespace    Foo {}
namespace    Bar {}',
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield 'braces with newlines' => [
            '<?php
namespace Foo {}
namespace Bar {}',
            '<?php
namespace
Foo {}
namespace
Bar {}',
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield 'with // comment' => [
            '<?php
namespace // comment
Foo;',
            null,
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield 'with /* comment */' => [
            '<?php
namespace /* comment */ Foo;',
            '<?php
namespace/* comment */ Foo;',
            ['constructs_followed_by_a_single_space' => ['namespace']],
        ];

        yield [
            '<?php
                switch ($a){ case 1: echo 123; }
                switch ($b){ case 1: echo 123; }
            ',
            '<?php
                switch($a){ case 1: echo 123; }
                switch  ($b){ case 1: echo 123; }
            ',
            ['constructs_followed_by_a_single_space' => ['switch']],
        ];

        yield [
            '<?php function foo(): array { return []; }',
            "<?php function foo():\narray { return []; }",
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];

        yield [
            '<?php interface F { public function foo(): array; }',
            "<?php interface F { public function foo():\tarray; }",
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];

        yield [
            '<?php $a=1; $f = function () use($a): array {};',
            '<?php $a=1; $f = function () use($a):array {};',
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];

        yield [
            '<?php fn()        : array => [];',
            '<?php fn()        :      array => [];',
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];

        yield [
            '<?php $a=1; $f = fn (): array => [];',
            '<?php $a=1; $f = fn ():      array => [];',
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];
    }

    /**
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, string $input): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<string, array{string, string}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield 'match 1' => [
            '<?php echo match ($x) {
    1, 2 => "Same for 1 and 2",
};',
            '<?php      echo              match     ($x) {
    1, 2 => "Same for 1 and 2",
};',
        ];

        yield 'match 2' => [
            '<?php echo match ($x) {
    1, 2 => "Same for 1 and 2",
};',
            '<?php echo match($x) {
    1, 2 => "Same for 1 and 2",
};',
        ];

        yield 'constructor property promotion' => [
            '<?php
class Point {
    public function __construct(
        public float $x = 0.0,
        protected float $y = 0.0,
        private float $z = 0.0,
    ) {}
}
',
            "<?php
class Point {
    public function __construct(
        public       float \$x = 0.0,
        protected\tfloat \$y = 0.0,
        private\nfloat \$z = 0.0,
    ) {}
}
",
        ];

        yield 'attribute' => [
            '<?php class Foo {
    #[Required] // foo
    public $bar1;

    #[Required]
    public $bar2;
}',
            '<?php class Foo {
    #[Required]// foo
    public $bar1;

    #[Required]
    public $bar2;
}',
        ];

        yield 'named argument' => [
            '<?php $foo(test: 1);',
            '<?php $foo(test:    1);',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, string $input, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{string, string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield 'readonly' => [
            '<?php
final class Foo
{
    public readonly string $prop;

    public function __construct(
        public readonly float $x = 0.0,
    ) {}
}
            ',
            '<?php
final class Foo
{
    public readonly  string $prop;

    public function __construct(
        public    readonly   float $x = 0.0,
    ) {}
}
            ',
        ];

        yield 'enum' => [
            '<?php
enum Suit {
    case Hearts;
}
',
            '<?php
enum     Suit {
    case Hearts;
}
',
        ];

        yield 'enum full caps' => [
            '<?php
ENUM Suit {
    case Hearts;
}
',
            '<?php
ENUM     Suit {
    case     Hearts;
}
',
        ];

        yield [
            '<?php class Foo
{
    final public const X = "foo";
}',
            '<?php class Foo
{
    final   public   const    X = "foo";
}',
        ];

        yield [
            '<?php
class Test {
    public function __construct(
        public $prop = new Foo,
    ) {}
}
',
            '<?php
class    Test {
    public     function    __construct(
        public    $prop = new     Foo,
    ) {}
}
',
        ];

        yield [
            '<?php enum Foo: int {}',
            "<?php enum Foo:\nint {}",
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];

        yield [
            '<?php enum Foo: string {}',
            '<?php enum Foo:string {}',
            ['constructs_followed_by_a_single_space' => ['type_colon']],
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFix84Cases
     *
     * @requires PHP 8.4
     */
    public function testFix84(string $expected, string $input, array $configuration = []): void
    {
        $this->testFix($expected, $input, $configuration);
    }

    /**
     * @return iterable<string, array{string, string}>
     */
    public static function provideFix84Cases(): iterable
    {
        yield 'asymmetric visibility' => [
            <<<'PHP'
                <?php class Foo
                {
                    public public(set) bool $a;
                    public protected(set) bool $b;
                    public private(set) bool $c;
                }
                PHP,
            <<<'PHP'
                <?php class Foo
                {
                    public public(set)         bool $a;
                    public protected(set)      bool $b;
                    public private(set)        bool $c;
                }
                PHP,
        ];
    }
}
