<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\ListNotation;

use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\ListNotation\ListSyntaxFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\ListNotation\ListSyntaxFixer>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\ListNotation\ListSyntaxFixer
 */
final class ListSyntaxFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield 'default' => [
            '<?php $a = [$a, $b] = $a; [$b] = $a;',
            '<?php $a = list($a, $b) = $a; [$b] = $a;',
        ];

        yield from self::getFixToShortSyntaxCases();

        foreach (self::getFixToShortSyntaxCases() as $label => $shortCase) {
            if ('messy comments case' === $label) {
                continue;
            }

            yield 'to long - '.$label => [$shortCase[1], $shortCase[0], ['syntax' => 'long']];
        }

        // the reverse of this is different because of all the comments and white space,
        // therefore we override with a similar case here
        yield 'comment case' => [
            '<?php
#
list(#
$a#
)#
=#
$a#
;#',
            '<?php
#
[#
$a#
]#
=#
$a#
;#',
            ['syntax' => 'long'],
        ];

        yield ['<?php

class Test
{
    public function updateAttributeKey($key, $value)
    {
        $this->{camel_case($attributes)}[$key] = $value;
    }
}',
            null,
            ['syntax' => 'long'],
        ];

        yield [
            '<?php [$b[$a]] = $foo();',
            null,
            ['syntax' => 'long'],
        ];

        yield [
            '<?php [$iHaveList => list($x, $y) = getList()];',
            '<?php [$iHaveList => [$x, $y] = getList()];',
            ['syntax' => 'long'],
        ];
    }

    /**
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, ?string $input = null): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<string, array{string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield 'simple 8.1' => [
            '<?php $a = _list(...);',
        ];
    }

    /**
     * @return iterable<array{string, string}>
     */
    private static function getFixToShortSyntaxCases(): iterable
    {
        yield [
            '<?php [$x] = $a;',
            '<?php list($x) = $a;',
        ];

        yield [
            '<?php [$a, $b, $c] = $array;',
            '<?php list($a, $b, $c) = $array;',
        ];

        yield [
            '<?php ["a" => $a, "b" => $b, "c" => $c] = $array;',
            '<?php list("a" => $a, "b" => $b, "c" => $c) = $array;',
        ];

        yield [
            '<?php
#
[//
    $x] =/**/$a?>',
            '<?php
#
list(//
    $x) =/**/$a?>',
        ];

        yield 'messy comments case' => [
            '<?php
#a
#g
[#h
#f
$a#
#e
]#
#
=#c
#
$a;#
#
',
            '<?php
#a
list#g
(#h
#f
$a#
#e
)#
#
=#c
#
$a;#
#
',
        ];

        yield [
            '<?php [$a, $b,, [$c, $d]] = $a;',
            '<?php list($a, $b,, list($c, $d)) = $a;',
        ];

        yield [
            '<?php [[$a, $b], [$c, $d]] = $a;',
            '<?php list(list($a, $b), list($c, $d)) = $a;',
        ];

        yield [
            '<?php [[$a, [$b]], [[$c, [$d]]]] = $a;',
            '<?php list(list($a, list($b)), list(list($c, list($d)))) = $a;',
        ];

        yield [
            '<?php [[$a]] = $foo();',
            '<?php list(list($a)) = $foo();',
        ];

        yield [
            '<?php foreach ($z as [$a, $b]) {}',
            '<?php foreach ($z as list($a, $b)) {}',
        ];

        yield [
            '<?php [$a, $b,, [$c, $d]] = $a;',
            '<?php list($a, $b,, list($c, $d)) = $a;',
        ];

        yield [
            '<?php [&$a, $b] = $a;',
            '<?php list(&$a, $b) = $a;',
        ];

        yield [
            '<?php [&$a,/* */&$b] = $a;',
            '<?php list(&$a,/* */&$b) = $a;',
        ];

        yield [
            '<?php [&$a, $b,, [&$c, $d]] = $a;',
            '<?php list(&$a, $b,, list(&$c, $d)) = $a;',
        ];
    }
}
