<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Operator;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Operator\ConcatSpaceFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Operator\ConcatSpaceFixer>
 *
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Operator\ConcatSpaceFixer
 */
final class ConcatSpaceFixerTest extends AbstractFixerTestCase
{
    /**
     * @dataProvider provideInvalidConfigurationCases
     *
     * @param _AutogeneratedInputConfiguration $configuration
     */
    public function testInvalidConfiguration(array $configuration, string $exceptionExpression): void
    {
        self::expectException(InvalidFixerConfigurationException::class);
        self::expectExceptionMessageMatches($exceptionExpression);

        $this->fixer->configure($configuration);
    }

    /**
     * @return iterable<string, array{array<array-key, mixed>, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'missing key' => [
            ['a' => 1],
            '#^\[concat_space\] Invalid configuration: The option "a" does not exist\. Defined options are: "spacing"\.$#',
        ];

        yield 'invalid value' => [
            ['spacing' => 'tabs'],
            '#^\[concat_space\] Invalid configuration: The option "spacing" with value "tabs" is invalid\. Accepted values are: "one", "none"\.$#',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input, array $configuration): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, null|string, _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php $foo = "a".\'b\'."c"."d".$e.($f + 1);',
            '<?php $foo = "a" . \'b\' ."c". "d" . $e.($f + 1);',
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $foo = 1 ."foo";',
            '<?php $foo = 1 . "foo";',
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $foo = "foo". 1;',
            '<?php $foo = "foo" . 1;',
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $foo = "a".
"b";',
            '<?php $foo = "a" .
"b";',
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar"
    ."baz";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar"
                     //test
                     ."baz";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar"
                     /* test */
                     ."baz";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar" //
    ."baz";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar" //
                            ."baz"//
                            ."cex"/**/
                            ."dev"/**  */
                            ."baz"      //
                            ."cex"      /**/
                            ."ewer23"           '.'
                            ."dev"      /**  */
                    ;',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar" //
    ."baz" /**/
    ."something";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar"
    ."baz".      //
    "something";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            '<?php $a = "foobar"
    ."baz".      /**  */
    "something";',
            null,
            ['spacing' => 'none'],
        ];

        yield [
            "<?php
                \$longString = '*'
                    .'*****'
                    .'*****'
                    .'*****'
                    // Comment about next line
                    .'*****'
                    // Other comment
                    .'*****';
                ",
            "<?php
                \$longString = '*'
                    . '*****'
                    .  '*****'
                    .   '*****'
                    // Comment about next line
                    .  '*****'
                    // Other comment
                    .  '*****';
                ",
            ['spacing' => 'none'],
        ];

        yield [
            '<?php
                    $a =   //
                    $c .   /**/
                    $d     #
                    . $e   /**  */
                    . $f . //
                    $z;
                ',
            '<?php
                    $a =   //
                    $c   .   /**/
                    $d     #
                    .   $e   /**  */
                    .   $f   . //
                    $z;
                ',
            ['spacing' => 'one'],
        ];

        yield [
            '<?php $foo = "a" . \'b\' . "c" . "d" . $e . ($f + 1);',
            '<?php $foo = "a" . \'b\' ."c". "d"    .  $e.($f + 1);',
            ['spacing' => 'one'],
        ];

        yield [
            '<?php $foo = "a" .
"b";',
            '<?php $foo = "a".
"b";',
            ['spacing' => 'one'],
        ];

        yield [
            '<?php $a = "foobar"
    . "baz";',
            '<?php $a = "foobar"
    ."baz";',
            ['spacing' => 'one'],
        ];

        yield [
            '<?php echo $a . $b;
                    echo $d . $e .   //
                        $f;
                    echo $a . $b?>
                 <?php
                    echo $c;
                ',
            '<?php echo $a.$b;
                    echo $d    .            $e          .   //
                        $f;
                    echo $a   .                  $b?>
                 <?php
                    echo $c;
                ',
            ['spacing' => 'one'],
        ];
    }
}
