<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Phpdoc;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Phpdoc\GeneralPhpdocTagRenameFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Phpdoc\GeneralPhpdocTagRenameFixer>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Phpdoc\GeneralPhpdocTagRenameFixer
 */
final class GeneralPhpdocTagRenameFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);

        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
        ];

        yield [
            '<?php
    /**
     * @inheritDoc
     * @inheritDoc
     * {@inheritDoc}
     * {@inheritDoc}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            [
                'replacements' => ['inheritDocs' => 'inheritDoc'],
            ],
        ];

        yield [
            '<?php
    /**
     * @inheritdoc
     * @inheritdoc
     * {@inheritdoc}
     * {@inheritdoc}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            [
                'fix_annotation' => true,
                'fix_inline' => true,
                'replacements' => ['inheritdocs' => 'inheritdoc'],
                'case_sensitive' => false,
            ],
        ];

        yield [
            '<?php
    /**
     * @inheritDoc
     * @inheritDoc
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            [
                'fix_inline' => false,
                'replacements' => ['inheritDocs' => 'inheritDoc'],
            ],
        ];

        yield [
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritDoc}
     * {@inheritDoc}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            [
                'fix_annotation' => false,
                'replacements' => ['inheritDocs' => 'inheritDoc'],
            ],
        ];

        yield [
            '<?php
    /**
     * @inheritdocs
     * @inheritDoc
     * {@inheritdocs}
     * {@inheritDoc}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            [
                'case_sensitive' => true,
                'replacements' => ['inheritDocs' => 'inheritDoc'],
            ],
        ];

        yield [
            '<?php
    /**
     * @inheritdoc
     * @inheritdoc
     * {@inheritdoc}
     * {@inheritdoc}
     * @see Foo::bar()
     * {@see Foo::bar()}
     */',
            '<?php
    /**
     * @inheritdocs
     * @inheritDocs
     * {@inheritdocs}
     * {@inheritDocs}
     * @link Foo::bar()
     * {@link Foo::bar()}
     */',
            [
                'replacements' => [
                    'inheritdocs' => 'inheritdoc',
                    'link' => 'see',
                ],
            ],
        ];

        yield [
            '<?php
    /**
     * @var int $foo
     * @Annotation("@type")
     */',
            '<?php
    /**
     * @type int $foo
     * @Annotation("@type")
     */',
            [
                'fix_annotation' => true,
                'fix_inline' => true,
                'replacements' => [
                    'type' => 'var',
                ],
            ],
        ];

        yield [
            '<?php
    /**
     * @var int $foo
     * @Annotation(\'@type\')
     * @Annotation("@type")
     */',
            '<?php
    /**
     * @type int $foo
     * @Annotation(\'@type\')
     * @Annotation("@type")
     */',
            [
                'fix_annotation' => true,
                'fix_inline' => false,
                'replacements' => [
                    'type' => 'var',
                ],
            ],
        ];
    }

    /**
     * @param array<string, mixed> $config
     *
     * @dataProvider provideInvalidConfigurationCases
     */
    public function testInvalidConfiguration(array $config, string $message): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);
        $this->expectExceptionMessage($message);

        $this->fixer->configure($config);
    }

    /**
     * @return iterable<array{mixed, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'invalid option' => [
            ['replacements' => true],
            '[general_phpdoc_tag_rename] Invalid configuration: The option "replacements" with value true is expected to be of type "string[]", but is of type "bool".',
        ];

        yield 'unknown option' => [
            ['foo' => true],
            '[general_phpdoc_tag_rename] Invalid configuration: The option "foo" does not exist. Defined options are: "case_sensitive", "fix_annotation", "fix_inline", "replacements".',
        ];

        yield [
            [
                'replacements' => [1 => 'abc'],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Tag to replace must be a string.',
        ];

        yield [
            [
                'replacements' => ['a' => null],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: The option "replacements" with value array is expected to be of type "string[]", but one of the elements is of type "null".',
        ];

        yield [
            [
                'replacements' => ['see' => 'link*/'],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Tag "see" cannot be replaced by invalid tag "link*/".',
        ];

        yield [
            [
                'replacements' => ['link' => 'see', 'a' => 'b', 'see' => 'link'],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Cannot change tag "link" to tag "see", as the tag "see" is configured to be replaced to "link".',
        ];

        yield [
            [
                'replacements' => ['b' => 'see', 'see' => 'link', 'link' => 'b'],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Cannot change tag "b" to tag "see", as the tag "see" is configured to be replaced to "link".',
        ];

        yield [
            [
                'replacements' => ['see' => 'link', 'link' => 'b'],
                'case_sensitive' => true,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Cannot change tag "see" to tag "link", as the tag "link" is configured to be replaced to "b".',
        ];

        yield [
            [
                'replacements' => ['Foo' => 'bar', 'foo' => 'baz'],
                'case_sensitive' => false,
            ],
            '[general_phpdoc_tag_rename] Invalid configuration: Tag "foo" cannot be configured to be replaced with several different tags when case sensitivity is off.',
        ];
    }
}
