<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Phpdoc;

use PhpCsFixer\ConfigurationException\InvalidConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;
use PhpCsFixer\WhitespacesFixerConfig;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Phpdoc\PhpdocAddMissingParamAnnotationFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Phpdoc\PhpdocAddMissingParamAnnotationFixer>
 *
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Phpdoc\PhpdocAddMissingParamAnnotationFixer
 */
final class PhpdocAddMissingParamAnnotationFixerTest extends AbstractFixerTestCase
{
    /**
     * @param array<array-key, mixed> $configuration
     *
     * @dataProvider provideInvalidConfigurationCases
     */
    public function testInvalidConfiguration(array $configuration, string $expectedMessage): void
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessageMatches($expectedMessage);

        $this->fixer->configure($configuration);
    }

    /**
     * @return iterable<string, array{mixed, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'unknown key' => [
            ['foo' => 'bar'],
            '#The option "foo" does not exist\.#',
        ];

        yield 'null' => [
            ['only_untyped' => null],
            '#expected to be of type "bool", but is of type "(null|NULL)"\.$#',
        ];

        yield 'int' => [
            ['only_untyped' => 1],
            '#expected to be of type "bool", but is of type "(int|integer)"\.$#',
        ];

        yield 'array' => [
            ['only_untyped' => []],
            '#expected to be of type "bool", but is of type "array"\.$#',
        ];

        yield 'float' => [
            ['only_untyped' => 0.1],
            '#expected to be of type "bool", but is of type "(float|double)"\.$#',
        ];

        yield 'object' => [
            ['only_untyped' => new \stdClass()],
            '#expected to be of type "bool", but is of type "stdClass"\.$#',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = [], bool $useTabsAndWindowsNewlines = false): void
    {
        $this->fixer->configure($configuration);
        if ($useTabsAndWindowsNewlines) {
            $this->fixer->setWhitespacesConfig(new WhitespacesFixerConfig("\t", "\r\n"));
        }
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php
    /**
     *
     */',
            null,
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $foo
     * @param mixed $bar
     */
    function f1($foo, $bar) {}',
            '<?php
    /**
     * @param int $foo
     */
    function f1($foo, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $bar
     * @param mixed $foo
     */
    function f2($foo, $bar) {}',
            '<?php
    /**
     * @param int $bar
     */
    function f2($foo, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @return void
     * @param mixed $foo
     * @param mixed $bar
     */
    function f3($foo, $bar) {}',
            '<?php
    /**
     * @return void
     */
    function f3($foo, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    abstract class Foo {
        /**
         * @param int $bar
         * @param mixed $foo
         */
        abstract public function f4a($foo, $bar);
    }',
            '<?php
    abstract class Foo {
        /**
         * @param int $bar
         */
        abstract public function f4a($foo, $bar);
    }',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    class Foo {
        /**
         * @param int $bar
         * @param mixed $foo
         */
        static final public function f4b($foo, $bar) {}
    }',
            '<?php
    class Foo {
        /**
         * @param int $bar
         */
        static final public function f4b($foo, $bar) {}
    }',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    class Foo {
        /**
         * @var int
         */
        private $foo;
    }',
            null,
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param $bar No type !!
     * @param mixed $foo
     */
    function f5($foo, $bar) {}',
            '<?php
    /**
     * @param $bar No type !!
     */
    function f5($foo, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int
     * @param int $bar
     * @param Foo\Bar $foo
     */
    function f6(Foo\Bar $foo, $bar) {}',
            '<?php
    /**
     * @param int
     * @param int $bar
     */
    function f6(Foo\Bar $foo, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $bar
     * @param null|string $foo
     */
    function f7(string $foo = nuLl, $bar) {}',
            '<?php
    /**
     * @param int $bar
     */
    function f7(string $foo = nuLl, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $bar
     * @param mixed $baz
     *
     * @return void
     */
    function f9(string $foo, $bar, $baz) {}',
            '<?php
    /**
     * @param int $bar
     *
     * @return void
     */
    function f9(string $foo, $bar, $baz) {}',
            ['only_untyped' => true],
        ];

        yield [
            '<?php
    /**
     * @param bool|bool[] $caseSensitive Line 1
     *                                   Line 2
     */
     function f11($caseSensitive) {}
',
            null,
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /** @return string */
    function hello($string)
    {
        return $string;
    }',
            null,
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /** @return string
     * @param mixed $string
     */
    function hello($string)
    {
        return $string;
    }',
            '<?php
    /** @return string
     */
    function hello($string)
    {
        return $string;
    }',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param mixed $string
     * @return string */
    function hello($string)
    {
        return $string;
    }',
            '<?php
    /**
     * @return string */
    function hello($string)
    {
        return $string;
    }',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $bar
     * @param string $foo
     */
    function f8(string $foo = "null", $bar) {}',
            '<?php
    /**
     * @param int $bar
     */
    function f8(string $foo = "null", $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @{inheritdoc}
     */
    function f10(string $foo = "null", $bar) {}',
        ];

        yield [
            '<?php
    /**
     * @inheritDoc
     */
    function f10(string $foo = "null", $bar) {}',
            null,
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * @param int $bar
     * @param ?array $foo
     */
    function p1(?array $foo = null, $bar) {}',
            '<?php
    /**
     * @param int $bar
     */
    function p1(?array $foo = null, $bar) {}',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
    /**
     * Foo
     * @param mixed $bar
     */
    function p1(?int $foo = 0, $bar) {}',
            '<?php
    /**
     * Foo
     */
    function p1(?int $foo = 0, $bar) {}',
            ['only_untyped' => true],
        ];

        yield [
            '<?php
    /**
     * Foo
     * @return int
     */
    function p1(?int $foo = 0) {}',
            null,
            ['only_untyped' => true],
        ];

        yield [
            "<?php\r\n\t/**\r\n\t * @param int \$bar\r\n\t * @param null|string \$foo\r\n\t */\r\n\tfunction f7(string \$foo = nuLl, \$bar) {}",
            "<?php\r\n\t/**\r\n\t * @param int \$bar\r\n\t */\r\n\tfunction f7(string \$foo = nuLl, \$bar) {}",
            ['only_untyped' => false],
            true,
        ];

        yield [
            '<?php
                    /**
                     * something
                     * @param mixed $numbers
                     */
                    function add(&$numbers) {}
                ',
            '<?php
                    /**
                     * something
                     */
                    function add(&$numbers) {}
                ',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
                    /**
                     * something
                     * @param null|array $numbers
                     */
                    function add(array &$numbers = null) {}
                ',
            '<?php
                    /**
                     * something
                     */
                    function add(array &$numbers = null) {}
                ',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
                    /**
                     * something
                     * @param array $numbers
                     */
                    function sum(...$numbers) {}
                ',
            '<?php
                    /**
                     * something
                     */
                    function sum(...$numbers) {}
                ',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
                    /**
                     * @param int $a
                     * @param array $numbers
                     */
                    function sum($a, ...$numbers) {}
                ',
            '<?php
                    /**
                     * @param int $a
                     */
                    function sum($a, ...$numbers) {}
                ',
            ['only_untyped' => false],
        ];

        yield [
            '<?php
                    /**
                     * @param \Date[] $numbers
                     */
                    function sum(\Date ...$numbers) {}
                ',
            '<?php
                    /**
                     */
                    function sum(\Date ...$numbers) {}
                ',
            ['only_untyped' => false],
        ];
    }

    /**
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, ?string $input = null): void
    {
        $this->fixer->configure(['only_untyped' => false]);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, string}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield [
            '<?php class Foo {
                /**
                 * @param Bar $x
                 * @param ?Bar $y
                 * @param null|Bar $z
                 */
                public function __construct(
                    public Bar $x,
                    protected ?Bar $y,
                    private null|Bar $z,
                ) {}
            }',
            '<?php class Foo {
                /**
                 */
                public function __construct(
                    public Bar $x,
                    protected ?Bar $y,
                    private null|Bar $z,
                ) {}
            }',
        ];
    }

    /**
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, ?string $input = null): void
    {
        $this->fixer->configure(['only_untyped' => false]);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield [
            '<?php class Foo {
                /**
                 * @param Bar $bar
                 * @param Baz $baz
                 */
                public function __construct(
                    public readonly Bar $bar,
                    readonly public Baz $baz,
                ) {}
            }',
            '<?php class Foo {
                /**
                 */
                public function __construct(
                    public readonly Bar $bar,
                    readonly public Baz $baz,
                ) {}
            }',
        ];
    }

    /**
     * @dataProvider provideFix84Cases
     *
     * @requires PHP 8.4
     */
    public function testFix84(string $expected, ?string $input = null): void
    {
        $this->testFix($expected, $input, ['only_untyped' => false]);
    }

    /**
     * @return iterable<string, array{string, string}>
     */
    public static function provideFix84Cases(): iterable
    {
        yield 'asymmetric visibility' => [
            <<<'PHP'
                <?php class Foo {
                    /**
                     * @param bool $a
                     * @param bool $b
                     * @param bool $c
                     */
                    public function __construct(
                        public public(set) bool $a,
                        public protected(set) bool $b,
                        public private(set) bool $c,
                    ) {}
                }
                PHP,
            <<<'PHP'
                <?php class Foo {
                    /**
                     */
                    public function __construct(
                        public public(set) bool $a,
                        public protected(set) bool $b,
                        public private(set) bool $c,
                    ) {}
                }
                PHP,
        ];
    }
}
