<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Phpdoc;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Phpdoc\PhpdocNoAliasTagFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Phpdoc\PhpdocNoAliasTagFixer>
 *
 * @author Graham Campbell <hello@gjcampbell.co.uk>
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Phpdoc\PhpdocNoAliasTagFixer
 */
final class PhpdocNoAliasTagFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $config
     *
     * @dataProvider provideInvalidConfigurationCases
     */
    public function testInvalidConfiguration(array $config, string $expectedMessage): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);
        $this->expectExceptionMessageMatches($expectedMessage);

        $this->fixer->configure($config);
    }

    /**
     * @return iterable<int, array{array<string, mixed>, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield [
            ['replacements' => [1 => 'abc']],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: Tag to replace must be a string\.$#',
        ];

        yield [
            ['replacements' => ['a' => null]],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: The option "replacements" with value array is expected to be of type "string\[\]", but one of the elements is of type "null"\.$#',
        ];

        yield [
            ['replacements' => ['see' => 'link*/']],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: Tag "see" cannot be replaced by invalid tag "link\*\/"\.$#',
        ];

        yield [
            ['foo' => 123],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: The option "foo" does not exist. Defined options are: "replacements"\.$#',
        ];

        yield [
            ['replacements' => [
                'link' => 'see',
                'a' => 'b',
                'see' => 'link',
            ]],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: Cannot change tag "link" to tag "see", as the tag "see" is configured to be replaced to "link"\.$#',
        ];

        yield [
            ['replacements' => [
                'b' => 'see',
                'see' => 'link',
                'link' => 'b',
            ]],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: Cannot change tag "b" to tag "see", as the tag "see" is configured to be replaced to "link"\.$#',
        ];

        yield [
            ['replacements' => [
                'see' => 'link',
                'link' => 'b',
            ]],
            '#^\[phpdoc_no_alias_tag\] Invalid configuration: Cannot change tag "see" to tag "link", as the tag "link" is configured to be replaced to "b"\.$#',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{0: string, 1?: ?string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php /** @see  https://github.com/php-fig/fig-standards/blob/master/proposed/phpdoc.md#710-link-deprecated */',
            '<?php /** @link  https://github.com/php-fig/fig-standards/blob/master/proposed/phpdoc.md#710-link-deprecated */',
        ];

        yield [
            '<?php /** @property mixed $bar */',
            '<?php /** @property-write mixed $bar */',
        ];

        yield [
            '<?php /** @property mixed $bar */',
            '<?php /** @property-read mixed $bar */',
        ];

        yield [
            '<?php /** @var string Hello! */',
            '<?php /** @type string Hello! */',
        ];

        yield [
            '<?php
    /**
     *
     */',
            null,
            ['replacements' => [
                'property-read' => 'property',
                'property-write' => 'property',
            ]],
        ];

        yield [
            '<?php
    /**
     * @property string $foo
     */',
            '<?php
    /**
     * @property-read string $foo
     */',
            ['replacements' => [
                'property-read' => 'property',
                'property-write' => 'property',
            ]],
        ];

        yield [
            '<?php /** @property mixed $bar */',
            '<?php /** @property-write mixed $bar */',
            ['replacements' => [
                'property-read' => 'property',
                'property-write' => 'property',
            ]],
        ];

        yield [
            '<?php
    /**
     *
     */',
            null,
            ['replacements' => [
                'type' => 'var',
            ]],
        ];

        yield [
            '<?php
    /**
     * @var string Hello!
     */',
            '<?php
    /**
     * @type string Hello!
     */',
            ['replacements' => [
                'type' => 'var',
            ]],
        ];

        yield [
            '<?php /** @var string Hello! */',
            '<?php /** @type string Hello! */',
            ['replacements' => [
                'type' => 'var',
            ]],
        ];

        yield [
            '<?php
    /**
     * Initializes this class with the given options.
     *
     * @param array $options {
     *     @var bool   $required Whether this element is required
     *     @var string $label    The display name for this element
     * }
     */',
            '<?php
    /**
     * Initializes this class with the given options.
     *
     * @param array $options {
     *     @type bool   $required Whether this element is required
     *     @type string $label    The display name for this element
     * }
     */',
            ['replacements' => [
                'type' => 'var',
            ]],
        ];

        yield [
            '<?php
    /**
     *
     */',
            null,
            ['replacements' => [
                'var' => 'type',
            ]],
        ];

        yield [
            '<?php
    /**
     * @type string Hello!
     */',
            '<?php
    /**
     * @var string Hello!
     */',
            ['replacements' => [
                'var' => 'type',
            ]],
        ];

        yield [
            '<?php /** @type string Hello! */',
            '<?php /** @var string Hello! */',
            ['replacements' => [
                'var' => 'type',
            ]],
        ];

        yield [
            '<?php
    /**
     * Initializes this class with the given options.
     *
     * @param array $options {
     *     @type bool   $required Whether this element is required
     *     @type string $label    The display name for this element
     * }
     */',
            '<?php
    /**
     * Initializes this class with the given options.
     *
     * @param array $options {
     *     @var bool   $required Whether this element is required
     *     @var string $label    The display name for this element
     * }
     */',
            ['replacements' => [
                'var' => 'type',
            ]],
        ];

        yield [
            '<?php /** @see  https://github.com/php-fig/fig-standards/blob/master/proposed/phpdoc.md#710-link-deprecated */',
            '<?php /** @link  https://github.com/php-fig/fig-standards/blob/master/proposed/phpdoc.md#710-link-deprecated */',
            ['replacements' => [
                'link' => 'see',
            ]],
        ];

        yield [
            <<<'PHP'
                <?php
                /**
                 * @see example.com
                 * @var foo
                 *
                 * @phpstan-type MyArray array{
                 *  '@link': ?string,
                 *  "@type": string,
                 * }
                 */
                class Foo {}
                PHP,
            <<<'PHP'
                <?php
                /**
                 * @link example.com
                 * @type foo
                 *
                 * @phpstan-type MyArray array{
                 *  '@link': ?string,
                 *  "@type": string,
                 * }
                 */
                class Foo {}
                PHP,
        ];
    }
}
