<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Phpdoc;

use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Phpdoc\PhpdocTypesOrderFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Phpdoc\PhpdocTypesOrderFixer>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Phpdoc\PhpdocTypesOrderFixer
 */
final class PhpdocTypesOrderFixerTest extends AbstractFixerTestCase
{
    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php /** @var null|int|string */',
            '<?php /** @var string|int|null */',
        ];

        yield [
            '<?php /** @param Bar|\Foo */',
            '<?php /** @param \Foo|Bar */',
        ];

        yield [
            '<?php /** @property-read \Bar|Foo */',
            '<?php /** @property-read Foo|\Bar */',
        ];

        yield [
            '<?php /** @property-write bar|Foo */',
            '<?php /** @property-write Foo|bar */',
        ];

        yield [
            '<?php /** @return Bar|foo */',
            '<?php /** @return foo|Bar */',
        ];

        yield [
            '<?php /** @method \Bar|Foo foo(\Bar|Foo $foo, \Bar|Foo $bar) */',
            '<?php /** @method Foo|\Bar foo(Foo|\Bar $foo, Foo|\Bar $bar) */',
        ];

        yield [
            '<?php /** @var null|array|Bar\Baz|bool[]|false|Foo|int|object|resource|string|string[] */',
            '<?php /** @var string[]|resource|false|object|null|Foo|Bar\Baz|bool[]|string|array|int */',
        ];

        yield [
            '<?php /** @var null|array<int, string> Foo */',
        ];

        yield [
            '<?php /** @var null|array<int, array<string>> Foo */',
        ];

        yield [
            '<?php /** @var NULL|int|string */',
            '<?php /** @var string|int|NULL */',
        ];

        yield [
            '<?php /** @var ?Bar|Foo */',
            '<?php /** @var Foo|?Bar */',
        ];

        yield [
            '<?php /** @var Bar|?Foo */',
            '<?php /** @var ?Foo|Bar */',
        ];

        yield [
            '<?php /** @var ?\Bar|Foo */',
            '<?php /** @var Foo|?\Bar */',
        ];

        yield [
            '<?php /** @var Bar|?\Foo */',
            '<?php /** @var ?\Foo|Bar */',
        ];

        yield [
            '<?php /** @var array<null|int|string> */',
            '<?php /** @var array<string|int|null> */',
        ];

        yield [
            '<?php /** @var array<int|string, null|int|string> */',
            '<?php /** @var array<string|int, string|int|null> */',
        ];

        yield [
            '<?php /** @var array<int|string,     array<int|string, null|int|string>> */',
            '<?php /** @var array<string|int,     array<string|int, string|int|null>> */',
        ];

        yield [
            '<?php /** @var null|null */',
        ];

        yield [
            '<?php /** @var null|\null */',
        ];

        yield [
            '<?php /** @var \null|null */',
        ];

        yield [
            '<?php /** @var \null|\null */',
        ];

        yield [
            '<?php /** @var array<\null|int|string> */',
            '<?php /** @var array<string|\null|int> */',
        ];

        yield [
            '<?php /** @var array<int, array<int, array<int, array<int, array<null|OutputInterface>>>>> */',
            '<?php /** @var array<int, array<int, array<int, array<int, array<OutputInterface|null>>>>> */',
        ];

        yield [
            '<?php /** @var null|Foo|Foo[]|Foo\Bar|Foo_Bar */',
            '<?php /** @var Foo[]|null|Foo|Foo\Bar|Foo_Bar */',
        ];

        yield [
            '<?php /** @return array<array<string, int>> */',
        ];

        yield [
            '<?php /** @return array<int, callable(array<string, null|string> , DateTime): bool> */',
        ];

        yield [
            '<?php /** @var null|string */',
            '<?php /** @var string|null */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @param null|string $foo */',
            '<?php /** @param string|null $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @property null|string $foo */',
            '<?php /** @property string|null $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @property-read null|string $foo */',
            '<?php /** @property-read string|null $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @property-write null|string $foo */',
            '<?php /** @property-write string|null $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @method null|string foo(null|int $foo, null|string $bar) */',
            '<?php /** @method string|null foo(int|null $foo, string|null $bar) */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return null|string */',
            '<?php /** @return string|null */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|string[]|resource|false|object|Foo|Bar\Baz|bool[]|string|array|int */',
            '<?php /** @var string[]|resource|false|object|null|Foo|Bar\Baz|bool[]|string|array|int */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|array<int, string> Foo */',
            '<?php /** @var array<int, string>|null Foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|array<int, array<string>> Foo */',
            '<?php /** @var array<int, array<string>>|null Foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var NULL|string */',
            '<?php /** @var string|NULL */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var Foo|?Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var ?Foo|Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<null|string> */',
            '<?php /** @var array<string|null> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<int, null|string> */',
            '<?php /** @var array<int, string|null> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<int,     array<null|int|string>> */',
            '<?php /** @var array<int,     array<int|string|null>> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|\null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var \null|null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var \null|\null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var \null|int */',
            '<?php /** @var int|\null */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<\null|int> */',
            '<?php /** @var array<int|\null> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<int, array<int, array<int, array<int, array<null|OutputInterface>>>>> */',
            '<?php /** @var array<int, array<int, array<int, array<int, array<OutputInterface|null>>>>> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var null|Foo[]|Foo|Foo\Bar|Foo_Bar */',
            '<?php /** @var Foo[]|null|Foo|Foo\Bar|Foo_Bar */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @method null|Y|X setOrder(array<string, null|array{Y,X,null|Z}> $by) */',
            '<?php /** @method Y|null|X setOrder(array<string, array{Y,X,Z|null}|null> $by) */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield '@method with invalid 2nd phpdoc' => [
            '<?php /** @method null|X setOrder($$by) */',
            '<?php /** @method X|null setOrder($$by) */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var array<array<int, int>, OutputInterface> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var iterable<array{names:array<string>, surname:string}> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var iterable<array{surname:string, names:array<string>}> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return array<array{level:string, message:string, context:array<mixed>}> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return Data<array{enabled: string[], all: array<string, string>}> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return array<int, callable(array<string, null|string> , DateTime): bool> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @param null|callable(array<string>): array<string, T> $callback */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return Closure(Iterator<TKey, T>): Generator<int, array<TKey, T>> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var Closure(Iterator<TKey, T>): Generator<int, array<TKey, T>> $pipe */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @return Generator<int, Promise<mixed>, mixed, Identity> */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @param null|callable(null|foo, null|bar): array<string, T> $callback */',
            '<?php /** @param null|callable(foo|null, bar|null): array<string, T> $callback */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @param null|string$foo */',
            '<?php /** @param string|null$foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_first'],
        ];

        yield [
            '<?php /** @var string|null */',
            '<?php /** @var null|string */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @param string|null $foo */',
            '<?php /** @param null|string $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @property string|null $foo */',
            '<?php /** @property null|string $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @property-read string|null $foo */',
            '<?php /** @property-read null|string $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @property-write string|null $foo */',
            '<?php /** @property-write null|string $foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @method string|null foo(int|null $foo, string|null $bar) */',
            '<?php /** @method null|string foo(null|int $foo, null|string $bar) */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @return string|null */',
            '<?php /** @return null|string */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var string[]|resource|false|object|Foo|Bar\Baz|bool[]|string|array|int|null */',
            '<?php /** @var string[]|resource|false|object|null|Foo|Bar\Baz|bool[]|string|array|int */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, string>|null Foo */',
            '<?php /** @var null|array<int, string> Foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, array<string>>|null Foo */',
            '<?php /** @var null|array<int, array<string>> Foo */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var string|NULL */',
            '<?php /** @var NULL|string */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Foo|?Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var ?Foo|Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Foo|?\Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var ?\Foo|Bar */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<string|null> */',
            '<?php /** @var array<null|string> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, string|null> */',
            '<?php /** @var array<int, null|string> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int,     array<int|string|null>> */',
            '<?php /** @var array<int,     array<null|int|string>> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var null|null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var null|\null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var \null|null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var \null|\null */',
            null,
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var int|\null */',
            '<?php /** @var \null|int */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int|\null> */',
            '<?php /** @var array<\null|int> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, array<int, array<int, array<int, array<OutputInterface|null>>>>> */',
            '<?php /** @var array<int, array<int, array<int, array<int, array<null|OutputInterface>>>>> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Foo[]|Foo|Foo\Bar|Foo_Bar|null */',
            '<?php /** @var Foo[]|null|Foo|Foo\Bar|Foo_Bar */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @return array<int, callable(array<string, string|null> , DateTime): bool> */',
            '<?php /** @return array<int, callable(array<string, null|string> , DateTime): bool> */',
            ['sort_algorithm' => 'none', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var int|null|string */',
            '<?php /** @var string|int|null */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param Bar|\Foo */',
            '<?php /** @param \Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @property-read \Bar|Foo */',
            '<?php /** @property-read Foo|\Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @property-write bar|Foo */',
            '<?php /** @property-write Foo|bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return Bar|foo */',
            '<?php /** @return foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @method \Bar|Foo foo(\Bar|Foo $foo, \Bar|Foo $bar) */',
            '<?php /** @method Foo|\Bar foo(Foo|\Bar $foo, Foo|\Bar $bar) */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array|Bar\Baz|bool[]|false|Foo|int|null|object|resource|string|string[] */',
            '<?php /** @var string[]|resource|false|object|null|Foo|Bar\Baz|bool[]|string|array|int */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int, string>|null Foo */',
            '<?php /** @var null|array<int, string> Foo */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int, array<string>>|null Foo */',
            '<?php /** @var null|array<int, array<string>> Foo */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var ?Bar|Foo */',
            '<?php /** @var Foo|?Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var Bar|?Foo */',
            '<?php /** @var ?Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var ?\Bar|Foo */',
            '<?php /** @var Foo|?\Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var Bar|?\Foo */',
            '<?php /** @var ?\Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<null|string> */',
            '<?php /** @var array<string|null> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int|string, null|string> */',
            '<?php /** @var array<string|int, string|null> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int|string,     array<int|string, null|string>> */',
            '<?php /** @var array<string|int,     array<string|int, string|null>> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var null|null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var null|\null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var \null|null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var \null|\null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var int|\null|string */',
            '<?php /** @var string|\null|int */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int|\null|string> */',
            '<?php /** @var array<string|\null|int> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var array<int, array<int, array<int, array<int, array<null|OutputInterface>>>>> */',
            '<?php /** @var array<int, array<int, array<int, array<int, array<OutputInterface|null>>>>> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var Foo|Foo[]|Foo\Bar|Foo_Bar|null */',
            '<?php /** @var Foo[]|null|Foo|Foo\Bar|Foo_Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return array<int, callable(array<string, null|string> , DateTime): bool> */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return A&B&C */',
            '<?php /** @return A&C&B */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return array<A&B&C> */',
            '<?php /** @return array<A&C&B> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return array<A&B&C>|bool|string */',
            '<?php /** @return bool|array<A&B&C>|string */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @return A&B<X|Y|Z>&C&D */',
            '<?php /** @return A&D&B<X|Y|Z>&C */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param A|(B&C) */',
            '<?php /** @param (C&B)|A */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param A|((A&B)|(B&C)) */',
            '<?php /** @param ((B&C)|(B&A))|A */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param A&(B&C) */',
            '<?php /** @param (C&B)&A */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param (A&C)|(B&C)|(C&D) */',
            '<?php /** @param (C&A)|(C&B)|(C&D) */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param \A|(\B&\C)|D */',
            '<?php /** @param D|\A|(\C&\B) */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @param A|((B&C)|D) */',
            '<?php /** @param (D|(C&B))|A */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var Closure<T>(T): T|null|string */',
            '<?php /** @var string|Closure<T>(T): T|null */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var \Closure<T of Model, T2, T3>(A|T, T3, T2): (T|T2)|null|string */',
            '<?php /** @var string|\Closure<T of Model, T2, T3>(T|A, T3, T2): (T2|T)|null */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var Closure<Closure_can_be_regular_class>|null|string */',
            '<?php /** @var string|Closure<Closure_can_be_regular_class>|null */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'none'],
        ];

        yield [
            '<?php /** @var int|string|null */',
            '<?php /** @var string|int|null */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @param Bar|\Foo */',
            '<?php /** @param \Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @property-read \Bar|Foo */',
            '<?php /** @property-read Foo|\Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @property-write bar|Foo */',
            '<?php /** @property-write Foo|bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @return Bar|foo */',
            '<?php /** @return foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @method \Bar|Foo foo(\Bar|Foo $foo, \Bar|Foo $bar) */',
            '<?php /** @method Foo|\Bar foo(Foo|\Bar $foo, Foo|\Bar $bar) */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array|Bar\Baz|bool[]|false|Foo|int|object|resource|string|string[]|null */',
            '<?php /** @var string[]|resource|false|object|null|Foo|Bar\Baz|bool[]|string|array|int */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, string>|null Foo */',
            '<?php /** @var null|array<int, string> Foo */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, array<string>>|null Foo */',
            '<?php /** @var null|array<int, array<string>> Foo */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var int|string|NULL */',
            '<?php /** @var string|int|NULL */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var ?Bar|Foo */',
            '<?php /** @var Foo|?Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Bar|?Foo */',
            '<?php /** @var ?Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var ?\Bar|Foo */',
            '<?php /** @var Foo|?\Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Bar|?\Foo */',
            '<?php /** @var ?\Foo|Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int|string|null> */',
            '<?php /** @var array<string|int|null> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int|string, int|string|null> */',
            '<?php /** @var array<string|int, string|int|null> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int|string,     array<int|string, int|string|null>> */',
            '<?php /** @var array<string|int,     array<string|int, string|int|null>> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var null|null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var null|\null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var \null|null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var \null|\null */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int|string|\null> */',
            '<?php /** @var array<string|\null|int> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var array<int, array<int, array<int, array<int, array<OutputInterface|null>>>>> */',
            '<?php /** @var array<int, array<int, array<int, array<int, array<null|OutputInterface>>>>> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var Foo|Foo[]|Foo\Bar|Foo_Bar|null */',
            '<?php /** @var Foo[]|null|Foo|Foo\Bar|Foo_Bar */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @return array<int, callable(array<string, string|null> , DateTime): bool> */',
            '<?php /** @return array<int, callable(array<string, null|string> , DateTime): bool> */',
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var ?Deferred<TestLocations> */',
            null,
            ['sort_algorithm' => 'alpha', 'null_adjustment' => 'always_last'],
        ];

        yield [
            '<?php /** @var AA|Aaa */',
            '<?php /** @var Aaa|AA */',
            ['case_sensitive' => true],
        ];
    }
}
