<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Whitespace;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;
use PhpCsFixer\WhitespacesFixerConfig;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Whitespace\BlankLineBeforeStatementFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Whitespace\BlankLineBeforeStatementFixer>
 *
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 * @author Andreas Möller <am@localheinz.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Whitespace\BlankLineBeforeStatementFixer
 */
final class BlankLineBeforeStatementFixerTest extends AbstractFixerTestCase
{
    /**
     * @dataProvider provideInvalidConfigurationCases
     *
     * @param mixed $controlStatement
     */
    public function testInvalidConfiguration($controlStatement): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);

        $this->fixer->configure([
            'statements' => [$controlStatement],
        ]);
    }

    /**
     * @return iterable<string, array{mixed}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'null' => [null];

        yield 'false' => [false];

        yield 'true' => [true];

        yield 'int' => [0];

        yield 'float' => [3.14];

        yield 'array' => [[]];

        yield 'object' => [new \stdClass()];

        yield 'unknown' => ['foo'];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = [], bool $useTabsAndWindowsNewlines = false): void
    {
        $this->fixer->configure($configuration);
        if ($useTabsAndWindowsNewlines) {
            $this->fixer->setWhitespacesConfig(new WhitespacesFixerConfig("\t", "\r\n"));
        }
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration, 3?: true}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php
if ($a) { /* 1 */ /* 2 */ /* 3 */ // something about $a
    return $b;
}
',
        ];

        yield [
            '<?php
if ($a) { // something about $a
    return $b;
}
',
        ];

        yield [
            '
$a = $a;
return $a;',
        ];

        yield [
            '<?php
$a = $a;

return $a;',
            '<?php
$a = $a; return $a;',
        ];

        yield [
            '<?php
$b = $b;

return $b;',
            '<?php
$b = $b;return $b;',
        ];

        yield [
            '<?php
$c = $c;

return $c;',
            '<?php
$c = $c;
return $c;',
        ];

        yield [
            '<?php
$d = $d;

return $d;',
            '<?php
$d = $d;
return $d;',
        ];

        yield [
            '<?php
if (true) {
    return 1;
}',
        ];

        yield [
            '<?php
if (true)
    return 1;',
        ];

        yield [
            '<?php
if (true) {
    return 1;
} else {
    return 2;
}',
        ];

        yield [
            '<?php
if (true)
    return 1;
else
    return 2;',
        ];

        yield [
            '<?php
if (true) {
    return 1;
} elseif (false) {
    return 2;
}',
        ];

        yield [
            '<?php
if (true)
    return 1;
elseif (false)
    return 2;',
        ];

        yield [
            '<?php
throw new Exception("return true; //.");',
        ];

        yield [
            '<?php
function foo()
{
    // comment
    return "foo";
}',
        ];

        yield [
            '<?php
function foo()
{
    // comment

    return "bar";
}',
        ];

        yield [
            '<?php

function foo()
{
    switch ($foo) {
        case 2: // comment
            return 1;
    }
}',
        ];

        yield 'do not fix when there is empty line between statement and preceding comment' => [
            '<?php function foo()
                {
                    bar();
                    // comment

                    return 42;
                }',
        ];

        yield 'do not fix when there is empty line between preceding comments' => [
            '<?php function foo()
                {
                    bar();
                    // comment1
                    // comment2

                    // comment3
                    return 42;
                }',
        ];

        yield [
            '<?php
switch ($a) {
    case 42:
        break;
}',
            null,
            ['statements' => ['break']],
        ];

        yield [
            '<?php
switch ($a) {
    case 42:
        $foo = $bar;

        break;
}',
            '<?php
switch ($a) {
    case 42:
        $foo = $bar;
        break;
}',
            ['statements' => ['break']],
        ];

        yield [
            '<?php
while (true) {
    if ($foo === $bar) {
        break;
    }
}',
            null,
            ['statements' => ['break']],
        ];

        yield [
            '<?php
while (true) {
    if ($foo === $bar) {
        break 1;
    }
}',
            null,
            ['statements' => ['break']],
        ];

        yield [
            '<?php
while (true) {
    if ($foo === $bar) {
        echo $baz;

        break;
    }
}',
            '<?php
while (true) {
    if ($foo === $bar) {
        echo $baz;
        break;
    }
}',
            ['statements' => ['break']],
        ];

        yield [
            '<?php
while (true) {
    if ($foo === $bar) {
        echo $baz;

        break 1;
    }
}',
            '<?php
while (true) {
    if ($foo === $bar) {
        echo $baz;
        break 1;
    }
}',
            ['statements' => ['break']],
        ];

        yield [
            '<?php
while (true) {
    if ($foo === $bar) {
        /** X */
        break 1;
    }
}',
            null,
            ['statements' => ['break']],
        ];

        yield [
            '<?php
switch ($a) {
    case 1:
        return 1;

    case 2;
        return 2;

    case 3:
        return 3;
}',
            '<?php
switch ($a) {
    case 1:
        return 1;
    case 2;
        return 2;
    case 3:
        return 3;
}',
            ['statements' => ['case']],
        ];

        yield [
            '<?php
while (true) {
    continue;
}',
            null,
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
while (true) {
    continue 1;
}',
            null,
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
while (true) {
    while (true) {
        continue 2;
    }
}',
            null,
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
while (true) {
    $foo = true;

    continue;
}',
            '<?php
while (true) {
    $foo = true;
    continue;
}',
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
while (true) {
    $foo = true;

    continue 1;
}',
            '<?php
while (true) {
    $foo = true;
    continue 1;
}',
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
while (true) {
    while (true) {
        switch($a) {
            case 1:
                echo 1;

                continue;
        }
        $foo = true;

        continue 2;
    }
}',
            '<?php
while (true) {
    while (true) {
        switch($a) {
            case 1:
                echo 1;
                continue;
        }
        $foo = true;
        continue 2;
    }
}',
            ['statements' => ['continue']],
        ];

        yield [
            '<?php
declare(ticks=1);',
            null,
            ['statements' => ['declare']],
        ];

        yield [
            '<?php
$foo = "bar";
do {
} while (true);
$foo = "bar";

declare(ticks=1);',
            '<?php
$foo = "bar";
do {
} while (true);
$foo = "bar";
declare(ticks=1);',
            ['statements' => ['declare']],
        ];

        yield [
            '<?php
switch ($a) {
    case 1:
        return 1;

    default:
        return 2;
}

switch ($a1) {
    default:
        return 22;
}',
            '<?php
switch ($a) {
    case 1:
        return 1;
    default:
        return 2;
}

switch ($a1) {
    default:
        return 22;
}',
            ['statements' => ['default']],
        ];

        yield [
            '<?php
do {
} while (true);',
            null,
            ['statements' => ['do']],
        ];

        yield [
            '<?php
$foo = "bar";

do {
} while (true);',
            '<?php
$foo = "bar";
do {
} while (true);',
            ['statements' => ['do']],
        ];

        yield [
            '<?php
if ($foo === $bar) {
    exit();
}',
            null,
            ['statements' => ['exit']],
        ];

        yield [
            '<?php
if ($foo === $bar) {
    echo $baz;

    exit();
}',
            '<?php
if ($foo === $bar) {
    echo $baz;
    exit();
}',
            ['statements' => ['exit']],
        ];

        yield [
            '<?php
if ($foo === $bar) {
    echo $baz;

    exit();
}',
            null,
            ['statements' => ['exit']],
        ];

        yield [
            '<?php
mysqli_connect() or exit();',
            null,
            ['statements' => ['exit']],
        ];

        yield [
            '<?php
if ($foo === $bar) {
    $bar = 9001;
    mysqli_connect() or exit();
}',
            null,
            ['statements' => ['exit']],
        ];

        yield [
            '<?php
                    echo 1;

                    for(;;){break;}
                ',
            '<?php
                    echo 1;
                    for(;;){break;}
                ',
            ['statements' => ['for']],
        ];

        yield [
            '<?php
a:

if ($foo === $bar) {
    goto a;
}',
            null,
            ['statements' => ['goto']],
        ];

        yield [
            '<?php
a:

if ($foo === $bar) {
    echo $baz;

    goto a;
}',
            '<?php
a:

if ($foo === $bar) {
    echo $baz;
    goto a;
}',
            ['statements' => ['goto']],
        ];

        yield [
            '<?php
a:

if ($foo === $bar) {
    echo $baz;

    goto a;
}',
            null,
            ['statements' => ['goto']],
        ];

        yield [
            '<?php if (true) {
    echo $bar;
}',
            null,
            ['statements' => ['if']],
        ];

        yield [
            '<?php
if (true) {
    echo $bar;
}',
            null,
            ['statements' => ['if']],
        ];

        yield [
            '<?php
$foo = $bar;

if (true) {
    echo $bar;
}',
            '<?php
$foo = $bar;
if (true) {
    echo $bar;
}',
            ['statements' => ['if']],
        ];

        yield [
            '<?php
// foo
if ($foo) { }',
            null,
            ['statements' => ['if']],
        ];

        yield [
            '<?php
                    echo 1;

                    foreach($a as $b){break;}
                ',
            '<?php
                    echo 1;
                    foreach($a as $b){break;}
                ',
            ['statements' => ['foreach']],
        ];

        yield [
            '<?php
include "foo.php";',
            null,
            ['statements' => ['include']],
        ];

        yield [
            '<?php
$foo = $bar;

include "foo.php";',
            '<?php
$foo = $bar;
include "foo.php";',
            ['statements' => ['include']],
        ];

        yield [
            '<?php
include_once "foo.php";',
            null,
            ['statements' => ['include_once']],
        ];

        yield [
            '<?php
$foo = $bar;

include_once "foo.php";',
            '<?php
$foo = $bar;
include_once "foo.php";',
            ['statements' => ['include_once']],
        ];

        yield [
            '<?php
require "foo.php";',
            null,
            ['statements' => ['require']],
        ];

        yield [
            '<?php
$foo = $bar;

require "foo.php";',
            '<?php
$foo = $bar;
require "foo.php";',
            ['statements' => ['require']],
        ];

        yield [
            '<?php
require_once "foo.php";',
            null,
            ['statements' => ['require_once']],
        ];

        yield [
            '<?php
$foo = $bar;

require_once "foo.php";',
            '<?php
$foo = $bar;
require_once "foo.php";',
            ['statements' => ['require_once']],
        ];

        yield [
            "<?php\r\n\$a = \$a;\r\n\r\nreturn \$a;",
            "<?php\r\n\$a = \$a; return \$a;",
            [],
            true,
        ];

        yield [
            "<?php\r\n\$b = \$b;\r\n\r\nreturn \$b;",
            "<?php\r\n\$b = \$b;return \$b;",
            [],
            true,
        ];

        yield [
            "<?php\r\n\$c = \$c;\r\n\r\nreturn \$c;",
            "<?php\r\n\$c = \$c;\r\nreturn \$c;",
            [],
            true,
        ];

        yield [
            '<?php
switch ($a) {
    case 42:
        break;
}',
            null,
            ['statements' => ['switch']],
        ];

        yield [
            '<?php
$foo = $bar;

switch ($foo) {
    case $bar:
        break;
}',
            '<?php
$foo = $bar;
switch ($foo) {
    case $bar:
        break;
}',
            ['statements' => ['switch']],
        ];

        yield [
            '<?php
if (false) {
    throw new \Exception("Something unexpected happened.");
}',
            null,
            ['statements' => ['throw']],
        ];

        yield [
            '<?php
if (false) {
    $log->error("No");

    throw new \Exception("Something unexpected happened.");
}',
            '<?php
if (false) {
    $log->error("No");
    throw new \Exception("Something unexpected happened.");
}',
            ['statements' => ['throw']],
        ];

        yield [
            '<?php
try {
    $transaction->commit();
} catch (\Exception $exception) {
    $transaction->rollback();
}',
            null,
            ['statements' => ['try']],
        ];

        yield [
            '<?php
$foo = $bar;

try {
    $transaction->commit();
} catch (\Exception $exception) {
    $transaction->rollback();
}',
            '<?php
$foo = $bar;
try {
    $transaction->commit();
} catch (\Exception $exception) {
    $transaction->rollback();
}',
            ['statements' => ['try']],
        ];

        yield [
            '<?php
while (true) {
    $worker->work();
}',
            null,
            ['statements' => ['while']],
        ];

        yield [
            '<?php
$foo = $bar;

while (true) {
    $worker->work();
}',
            '<?php
$foo = $bar;
while (true) {
    $worker->work();
}',
            ['statements' => ['while']],
        ];

        yield [
            '<?php
$foo = $bar;

do {
    echo 1;

    while($a());
    $worker->work();
} while (true);',
            '<?php
$foo = $bar;

do {
    echo 1;
    while($a());
    $worker->work();
} while (true);',
            ['statements' => ['while']],
        ];

        yield [
            '<?php
function foo() {
yield $a; /* a *//* b */     /* c */       /* d *//* e *//* etc */
   '.'
yield $b;
}',
            '<?php
function foo() {
yield $a; /* a *//* b */     /* c */       /* d *//* e *//* etc */   '.'
yield $b;
}',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    yield $a; // test

    yield $b; // test /* A */

    yield $c;

    yield $d;

yield $e;#

yield $f;

    /* @var int $g */
    yield $g;

/* @var int $h */
yield $i;

yield $j;
}',
            '<?php
function foo() {
    yield $a; // test
    yield $b; // test /* A */
    yield $c;
    yield $d;yield $e;#
yield $f;
    /* @var int $g */
    yield $g;
/* @var int $h */
yield $i;
yield $j;
}',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    yield $a;
}',
            null,
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    yield $a;

    yield $b;
}',
            '<?php
function foo() {
    yield $a;
    yield $b;
}',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    yield \'b\' => $a;

    yield "a" => $b;
}',
            '<?php
function foo() {
    yield \'b\' => $a;
    yield "a" => $b;
}',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    $a = $a;

    yield $a;
}',
            null,
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    $a = $a;

    yield $a;
}',
            '<?php
function foo() {
    $a = $a;
    yield $a;
}',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php function foo() {
                    // yield 1
                    yield 1;

                    // yield 2
                    yield 2;
                }',
            '<?php function foo() {
                    // yield 1
                    yield 1;
                    // yield 2
                    yield 2;
                }',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php function foo() {
                    yield 1;

                    // yield 2
                    // or maybe yield 3
                    // better compromise
                    yield 2.5;
                }',
            '<?php function foo() {
                    yield 1;
                    // yield 2
                    // or maybe yield 3
                    // better compromise
                    yield 2.5;
                }',
            ['statements' => ['yield']],
        ];

        yield [
            '<?php
function foo() {
    yield from $a;
}',
            null,
            ['statements' => ['yield_from']],
        ];

        yield [
            '<?php
function foo() {
    yield from $a;

    yield from $b;
}',
            '<?php
function foo() {
    yield from $a;
    yield from $b;
}',
            ['statements' => ['yield_from']],
        ];

        yield [
            '<?php
function foo() {
    $a = $a;

    yield from $a;

    yield $a;
    yield $b;
}',
            null,
            ['statements' => ['yield_from']],
        ];

        yield [
            '<?php
function foo() {
    $a = $a;

    yield from $a;
}',
            '<?php
function foo() {
    $a = $a;
    yield from $a;
}',
            ['statements' => ['yield_from']],
        ];

        $statementsWithoutCaseOrDefault = [
            'break',
            'continue',
            'declare',
            'do',
            'for',
            'foreach',
            'if',
            'include',
            'include_once',
            'require',
            'require_once',
            'return',
            'switch',
            'throw',
            'try',
            'while',
        ];

        $allStatements = [...$statementsWithoutCaseOrDefault, 'case', 'default'];

        yield [
            '<?php
                    while($a) {
                        if ($c) {
                            switch ($d) {
                                case $e:
                                    continue 2;
                                case 4:
                                    break;
                                case 5:
                                    return 1;
                                default:
                                    return 0;
                            }
                        }
                    }
                ',
            null,
            ['statements' => $statementsWithoutCaseOrDefault],
        ];

        yield [
            '<?php
                    while($a) {
                        if ($c) {
                            switch ($d) {
                                case $e:
                                    continue 2;

                                case 4:
                                    break;

                                case 5:
                                    return 1;

                                default:
                                    return 0;
                            }
                        }
                    }
                ',
            null,
            ['statements' => $allStatements],
        ];

        yield [
            '<?php
do {
    echo 0;
    if ($a) {
        echo 1;

        break;
    }
    echo 2;

    throw $f;
} while(true);',
            '<?php
do {
    echo 0;
    if ($a) {
        echo 1;
        break;
    }
    echo 2;
    throw $f;
} while(true);',
            ['statements' => ['break', 'throw']],
        ];

        yield [
            '<?php
/** @var int $foo */
$foo = 123;

/** @var float $bar */
$bar = 45.6;

/** @var string */
$baz = "789";
',
            '<?php
/** @var int $foo */
$foo = 123;
/** @var float $bar */
$bar = 45.6;
/** @var string */
$baz = "789";
',
            ['statements' => ['phpdoc']],
        ];

        yield [
            '<?php
/* header */

/**
 * Class description
 */
class Foo {
    /** test */
    public function bar() {}
}
',
            null,
            ['statements' => ['phpdoc']],
        ];
    }

    /**
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, ?string $input = null): void
    {
        $this->testFix($expected, $input, ['statements' => ['default']]);
    }

    /**
     * @return iterable<string, array{string}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield 'match' => [
            '<?php
                match ($foo) {
                    1 => "a",
                    default => "b"
                };

                match ($foo) {
                    1 => "a1",


                    default => "b2"
                };
            ',
        ];
    }

    /**
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, ?string $input = null): void
    {
        $this->testFix($expected, $input, ['statements' => ['case']]);
    }

    /**
     * @return iterable<string, array{string, string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield 'enum' => [
            '<?php
enum Suit {
    case Hearts;
    case Diamonds;
    case Clubs;


    case Spades;
}

enum UserStatus: string {
    case Pending = "P";
    case Active = "A";

    public function label(): string {
        switch ($a) {
            case 1:
                return 1;

            case 2:
                return 2; // do fix
        }

        return "label";
    }
}
',
            '<?php
enum Suit {
    case Hearts;
    case Diamonds;
    case Clubs;


    case Spades;
}

enum UserStatus: string {
    case Pending = "P";
    case Active = "A";

    public function label(): string {
        switch ($a) {
            case 1:
                return 1;
            case 2:
                return 2; // do fix
        }

        return "label";
    }
}
',
        ];
    }
}
