<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Whitespace;

use PhpCsFixer\ConfigurationException\InvalidFixerConfigurationException;
use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Whitespace\SpacesInsideParenthesesFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Whitespace\SpacesInsideParenthesesFixer>
 *
 * @author Marc Aubé
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Whitespace\SpacesInsideParenthesesFixer
 */
final class SpacesInsideParenthesesFixerTest extends AbstractFixerTestCase
{
    /**
     * @param array<string, mixed> $wrongConfig
     *
     * @dataProvider provideInvalidConfigurationCases
     */
    public function testInvalidConfiguration(array $wrongConfig, string $expectedMessage): void
    {
        $this->expectException(InvalidFixerConfigurationException::class);
        $this->expectExceptionMessageMatches($expectedMessage);

        $this->fixer->configure($wrongConfig);
    }

    /**
     * @return iterable<string, array{array<array-key, mixed>, string}>
     */
    public static function provideInvalidConfigurationCases(): iterable
    {
        yield 'missing key' => [
            ['a' => 1],
            '#^\[spaces_inside_parentheses\] Invalid configuration: The option "a" does not exist\. Defined options are: "space"\.$#',
        ];

        yield 'invalid value' => [
            ['space' => 'double'],
            '#^\[spaces_inside_parentheses\] Invalid configuration: The option "space" with value "double" is invalid\. Accepted values are: "none", "single"\.$#',
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<array{0: string, 1?: null|string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFixCases(): iterable
    {
        // default leaves new lines alone
        yield [
            "<?php

class Foo
{
    private function bar()
    {
        if (foo(
            'foo' ,
            'bar'    ,
            [1, 2, 3],
            'baz' // a comment just to mix things up
        )) {
            return 1;
        };
    }
}
",
        ];

        yield [
            '<?php foo();',
            '<?php foo( );',
        ];

        yield [
            '<?php
if (true) {
    // if body
}',
            '<?php
if ( true ) {
    // if body
}',
        ];

        yield [
            '<?php
if (true) {
    // if body
}',
            '<?php
if (     true   ) {
    // if body
}',
        ];

        yield [
            '<?php
function foo($bar, $baz)
{
    // function body
}',
            '<?php
function foo( $bar, $baz )
{
    // function body
}',
        ];

        yield [
            '<?php
$foo->bar($arg1, $arg2);',
            '<?php
$foo->bar(  $arg1, $arg2   );',
        ];

        yield [
            '<?php
$var = array( 1, 2, 3 );
',
        ];

        yield [
            '<?php
$var = [ 1, 2, 3 ];
',
        ];

        // list call with trailing comma - need to leave alone
        yield [
            '<?php list($path, $mode, ) = foo();',
        ];

        yield [
            '<?php list($path, $mode,) = foo();',
        ];

        yield [
            '<?php
$a = $b->test(  // do not remove space
    $e          // between `(` and `)`
                // and this comment
);',
        ];

        yield [
            '<?php
function hello($value) {
    // code...
}',
            '<?php
function hello( $value ) {
    // code...
}',
        ];

        yield [
            '<?php
$code = function ($hello, $there) use ($ami, $tumi) {
    // code
};
',
            '<?php
$code = function ( $hello, $there   ) use ( $ami, $tumi ) {
    // code
};
',
        ];

        yield [
            '<?php
for ($i = 0; $i < 42; $i++) {
    // code...
}
',
            '<?php
for (   $i = 0; $i < 42; $i++ ) {
    // code...
}
',
        ];

        yield [
            '<?php
explode($a, $b);
',
            '<?php
explode( $a, $b );
',
        ];

        yield [
            '<?php
if ($something) {
    // code
}
',
            '<?php
if (  $something      ) {
    // code
}
',
        ];

        yield [
            '<?php
multiply((2 + 3) * 4);
',
            '<?php
multiply( (    2 + 3  ) * 4    );
',
        ];

        yield [
            '<?php $x = (new Foo())->bar();',
            '<?php $x = ( new Foo() )->bar();',
        ];

        yield [
            '<?php $x = (new Foo)->bar;',
            '<?php $x = ( new Foo )->bar;',
        ];

        yield 'leaves new lines alone' => [
            "<?php

class Foo
{
    private function bar()
    {
        if ( foo(
            'foo' ,
            'bar'    ,
            [1, 2, 3],
            'baz' // a comment just to mix things up
        ) ) {
            return 1;
        };
    }
}",
            null,
            ['space' => 'single'],
        ];

        yield [
            '<?php foo();',
            '<?php foo( );',
            ['space' => 'single'],
        ];

        yield [
            '<?php
if ( true ) {
    // if body
}',
            '<?php
if (true) {
    // if body
}',
            ['space' => 'single'],
        ];

        yield [
            '<?php
if ( true ) {
    // if body
}',
            '<?php
if (     true   ) {
    // if body
}',
            ['space' => 'single'],
        ];

        yield [
            '<?php
function foo( $bar, $baz )
{
    // function body
}',
            '<?php
function foo($bar, $baz)
{
    // function body
}',
            ['space' => 'single'],
        ];

        yield [
            '<?php
$foo->bar( $arg1, $arg2 );',
            '<?php
$foo->bar(  $arg1, $arg2   );',
            ['space' => 'single'],
        ];

        yield [
            '<?php
$var = array( 1, 2, 3 );
',
            '<?php
$var = array(1, 2, 3);
',
            ['space' => 'single'],
        ];

        yield [
            '<?php
$var = [ 1, 2, 3 ];
',
            null,
            ['space' => 'single'],
        ];

        yield [
            '<?php list( $path, $mode, ) = foo();',
            '<?php list($path, $mode,) = foo();',
            ['space' => 'single'],
        ];

        yield [
            '<?php
$a = $b->test(  // do not remove space
    $e          // between `(` and `)`
                // and this comment
 );',
            null,
            ['space' => 'single'],
        ];

        yield [
            '<?php
function hello( $value ) {
    // code...
}',
            '<?php
function hello($value) {
    // code...
}',
            ['space' => 'single'],
        ];

        yield [
            '<?php
$code = function ( $hello, $there ) use ( $ami, $tumi ) {
    // code
};
',
            '<?php
$code = function ($hello, $there) use ($ami, $tumi) {
    // code
};
',
            ['space' => 'single'],
        ];

        yield [
            '<?php
for ( $i = 0; $i < 42; $i++ ) {
    // code...
}
',
            '<?php
for ($i = 0; $i < 42; $i++) {
    // code...
}
',
            ['space' => 'single'],
        ];

        yield [
            '<?php
explode( $a, $b );
',
            '<?php
explode($a, $b);
',
            ['space' => 'single'],
        ];

        yield [
            '<?php
if ( $something ) {
    // code
}
',
            '<?php
if (    $something    ) {
    // code
}
',
            ['space' => 'single'],
        ];

        yield [
            '<?php
multiply( ( 2 + 3 ) * 4 );
',
            '<?php
multiply((2 + 3) * 4);
',
            ['space' => 'single'],
        ];

        yield [
            '<?php $x = ( new Foo() )->bar();',
            '<?php $x = (new Foo())->bar();',
            ['space' => 'single'],
        ];
    }

    /**
     * @param _AutogeneratedInputConfiguration $configuration
     *
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, string $input, array $configuration = []): void
    {
        $this->fixer->configure($configuration);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<string, array{0: string, 1?: string, 2?: _AutogeneratedInputConfiguration}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield 'mixed argument' => [
            '<?php function foo(mixed $a){}',
            '<?php function foo( mixed $a ){}',
        ];

        yield 'mixed argument single space' => [
            '<?php function foo( mixed $a ){}',
            '<?php function foo(mixed $a){}',
            ['space' => 'single'],
        ];
    }
}
