<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Tests\Fixer\Whitespace;

use PhpCsFixer\Tests\Test\AbstractFixerTestCase;

/**
 * @internal
 *
 * @covers \PhpCsFixer\Fixer\Whitespace\TypeDeclarationSpacesFixer
 *
 * @extends AbstractFixerTestCase<\PhpCsFixer\Fixer\Whitespace\TypeDeclarationSpacesFixer>
 *
 * @author Dariusz Rumiński <dariusz.ruminski@gmail.com>
 * @author John Paul E. Balandan, CPA <paulbalandan@gmail.com>
 *
 * @phpstan-import-type _AutogeneratedInputConfiguration from \PhpCsFixer\Fixer\Whitespace\TypeDeclarationSpacesFixer
 */
final class TypeDeclarationSpacesFixerTest extends AbstractFixerTestCase
{
    /**
     * @dataProvider provideFixCases
     */
    public function testFix(string $expected, ?string $input = null): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, 1?: ?string}>
     */
    public static function provideFixCases(): iterable
    {
        yield [
            '<?php function foo(bool /**bla bla*/$param) {}',
            '<?php function foo(bool/**bla bla*/$param) {}',
        ];

        yield [
            '<?php function foo(bool /**bla bla*/$param) {}',
            '<?php function foo(bool  /**bla bla*/$param) {}',
        ];

        yield [
            '<?php function foo(callable $param) {}',
            '<?php function foo(callable$param) {}',
        ];

        yield [
            '<?php function foo(callable $param) {}',
            '<?php function foo(callable  $param) {}',
        ];

        yield [
            '<?php function foo(array &$param) {}',
            '<?php function foo(array&$param) {}',
        ];

        yield [
            '<?php function foo(array &$param) {}',
            '<?php function foo(array  &$param) {}',
        ];

        yield [
            '<?php function foo(array & $param) {}',
            '<?php function foo(array& $param) {}',
        ];

        yield [
            '<?php function foo(array & $param) {}',
            '<?php function foo(array  & $param) {}',
        ];

        yield [
            '<?php function foo(Bar $param) {}',
            '<?php function foo(Bar$param) {}',
        ];

        yield [
            '<?php function foo(Bar $param) {}',
            '<?php function foo(Bar  $param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz $param) {}',
            '<?php function foo(Bar\Baz$param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz $param) {}',
            '<?php function foo(Bar\Baz  $param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz &$param) {}',
            '<?php function foo(Bar\Baz&$param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz &$param) {}',
            '<?php function foo(Bar\Baz  &$param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz & $param) {}',
            '<?php function foo(Bar\Baz& $param) {}',
        ];

        yield [
            '<?php function foo(Bar\Baz & $param) {}',
            '<?php function foo(Bar\Baz  & $param) {}',
        ];

        yield [
            '<?php $foo = function(Bar\Baz $param) {};',
            '<?php $foo = function(Bar\Baz$param) {};',
        ];

        yield [
            '<?php $foo = function(Bar\Baz $param) {};',
            '<?php $foo = function(Bar\Baz  $param) {};',
        ];

        yield [
            '<?php $foo = function(Bar\Baz &$param) {};',
            '<?php $foo = function(Bar\Baz&$param) {};',
        ];

        yield [
            '<?php $foo = function(Bar\Baz &$param) {};',
            '<?php $foo = function(Bar\Baz  &$param) {};',
        ];

        yield [
            '<?php $foo = function(Bar\Baz & $param) {};',
            '<?php $foo = function(Bar\Baz& $param) {};',
        ];

        yield [
            '<?php $foo = function(Bar\Baz & $param) {};',
            '<?php $foo = function(Bar\Baz  & $param) {};',
        ];

        yield [
            '<?php class Test { public function foo(Bar\Baz $param) {} }',
            '<?php class Test { public function foo(Bar\Baz$param) {} }',
        ];

        yield [
            '<?php class Test { public function foo(Bar\Baz $param) {} }',
            '<?php class Test { public function foo(Bar\Baz  $param) {} }',
        ];

        yield [
            '<?php $foo = function(array $a,
                    array $b, array $c, array $d) {};',
            '<?php $foo = function(array $a,
                    array$b, array     $c, array
                    $d) {};',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz $param) => null;',
            '<?php $foo = fn(Bar\Baz$param) => null;',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz $param) => null;',
            '<?php $foo = fn(Bar\Baz  $param) => null;',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz &$param) => null;',
            '<?php $foo = fn(Bar\Baz&$param) => null;',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz &$param) => null;',
            '<?php $foo = fn(Bar\Baz  &$param) => null;',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz & $param) => null;',
            '<?php $foo = fn(Bar\Baz& $param) => null;',
        ];

        yield [
            '<?php $foo = fn(Bar\Baz & $param) => null;',
            '<?php $foo = fn(Bar\Baz  & $param) => null;',
        ];

        yield [
            '<?php $foo = fn(array $a,
                    array $b, array $c, array $d) => null;',
            '<?php $foo = fn(array $a,
                    array$b, array     $c, array
                    $d) => null;',
        ];

        yield [
            '<?php function foo(array ...$param) {}',
            '<?php function foo(array...$param) {}',
        ];

        yield [
            '<?php function foo(array & ...$param) {}',
            '<?php function foo(array& ...$param) {}',
        ];

        yield [
            '<?php class Foo { public int $x; }',
            '<?php class Foo { public int$x; }',
        ];

        yield [
            '<?php class Foo { public bool $x; }',
            '<?php class Foo { public bool    $x; }',
        ];

        yield [
            '<?php class Foo { protected \Bar\Baz $c; }',
            '<?php class Foo { protected \Bar\Baz$c; }',
        ];

        yield [
            '<?php class Foo { protected \Bar\Baz $c; }',
            '<?php class Foo { protected \Bar\Baz   $c; }',
        ];

        yield [
            '<?php class Foo { private array $x; }',
            '<?php class Foo { private array$x; }',
        ];

        yield [
            '<?php class Foo { private array $x; }',
            '<?php class Foo { private array
$x; }',
        ];

        yield [
            '<?php
class Point
{
    public \DateTime $x;
    protected bool $y = true;
    private array $z = [];
    public int $a = 0;
    protected string $b = \'\';
    private float $c = 0.0;
}
',
            '<?php
class Point
{
    public \DateTime    $x;
    protected bool      $y = true;
    private array       $z = [];
    public int          $a = 0;
    protected string    $b = \'\';
    private float       $c = 0.0;
}
',
        ];

        yield [
            '<?php function foo($param) {}',
        ];

        yield [
            '<?php function foo($param1,$param2) {}',
        ];

        yield [
            '<?php function foo(&$param) {}',
        ];

        yield [
            '<?php function foo(& $param) {}',
        ];

        yield [
            '<?php function foo(/**int*/$param) {}',
        ];

        yield [
            '<?php function foo(bool /**bla bla*/ $param) {}',
        ];

        yield [
            '<?php $foo = function(
                    array $a,
                    $b
                ) {};',
        ];

        yield [
            '<?php $foo = function(
                    $a,
                    array $b
                ) {};',
        ];

        yield [
            '<?php function foo(...$param) {}',
        ];

        yield [
            '<?php function foo(&...$param) {}',
        ];

        yield [
            '<?php use function some\test\{fn_a, fn_b, fn_c};',
        ];

        yield [
            '<?php use function some\test\{fn_a, fn_b, fn_c} ?>',
        ];

        yield [
            '<?php $foo = fn(
                    array $a,
                    $b
                ) => null;',
        ];

        yield [
            '<?php $foo = fn(
                    $a,
                    array $b
                ) => null;',
        ];

        yield [
            '<?php class Foo { public $p; }',
        ];

        yield [
            '<?php class Foo { protected /* int */ $a; }',
        ];

        yield [
            '<?php class Foo { private int $a; }',
        ];
    }

    /**
     * @dataProvider provideFix80Cases
     *
     * @requires PHP 8.0
     */
    public function testFix80(string $expected, string $input): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, 1?: ?string}>
     */
    public static function provideFix80Cases(): iterable
    {
        yield [
            '<?php function foo(mixed $a) {}',
            '<?php function foo(mixed$a) {}',
        ];

        yield [
            '<?php function foo(mixed $a) {}',
            '<?php function foo(mixed    $a) {}',
        ];

        yield [
            '<?php
class Foo
{
    public function __construct(
        public int $a,
        protected bool $b,
        private Bar\Baz $c,
    ) {}
}
',
            '<?php
class Foo
{
    public function __construct(
        public int  $a,
        protected bool$b,
        private Bar\Baz     $c,
    ) {}
}
',
        ];
    }

    /**
     * @dataProvider provideFix81Cases
     *
     * @requires PHP 8.1
     */
    public function testFix81(string $expected, ?string $input = null): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, 1?: ?string}>
     */
    public static function provideFix81Cases(): iterable
    {
        yield [
            '<?php class Foo { private readonly int $bar; }',
            '<?php class Foo { private readonly int$bar; }',
        ];

        yield [
            '<?php class Foo { private readonly int $bar; }',
            '<?php class Foo { private readonly int    $bar; }',
        ];

        yield [
            '<?php class Foo { readonly int $i; }',
            '<?php class Foo { readonly int$i; }',
        ];
    }

    /**
     * @dataProvider provideFix82Cases
     *
     * @requires PHP 8.2
     */
    public function testFix82(string $expected, ?string $input = null): void
    {
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, 1?: ?string}>
     */
    public static function provideFix82Cases(): iterable
    {
        yield [
            '<?php class Foo { public (A&B)|C $bar; }',
            '<?php class Foo { public (A&B)|C$bar; }',
        ];

        yield [
            '<?php class Foo { public (A&B)|C $bar; }',
            '<?php class Foo { public (A&B)|C    $bar; }',
        ];
    }

    /**
     * @dataProvider provideFix83Cases
     *
     * @requires PHP 8.3
     */
    public function testFix83(string $expected, ?string $input = null): void
    {
        $this->fixer->configure(['elements' => ['property', 'constant', 'function']]);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<int, array{string, 1?: ?string}>
     */
    public static function provideFix83Cases(): iterable
    {
        yield [
            '<?php class Foo {const   BAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const    BAR = 123; }',
        ];

        yield [
            '<?php class Foo {const BAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const string BAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const stringBAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const BAR = 123; }',
        ];

        yield [
            '<?php class Foo {public const array BAR = []; }',
            '<?php class Foo {public const array               BAR = []; }',
        ];

        yield [
            '<?php class Foo {public const bool BAR = true; }',
            '<?php class Foo {public const bool               BAR = true; }',
        ];

        yield [
            '<?php class Foo {public int $bar; public const string BAZ = ""; }',
            '<?php class Foo {public int   $bar; public const string               BAZ = ""; }',
        ];

        yield [
            '<?php class Foo {public const string BAR = ""; }',
            '<?php class Foo {public const string	BAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const string BAR = ""; }',
            '<?php class Foo {public const string	 	  BAR = ""; }',
        ];

        yield [
            '<?php class Foo {public const int BAR = 0; }',
            '<?php class Foo {public const int    BAR = 0; }',
        ];

        yield [
            '<?php class Bar {public const bool BAR = true; }',
            '<?php class Bar {public const bool    BAR = true; }',
        ];

        yield [
            '<?php interface Bar {public const string BAR = "value"; }',
            '<?php interface Bar {public const string    BAR = "value"; }',
        ];

        yield [
            '<?php trait Bar {public const string FOO = ""; }',
            '<?php trait Bar {public const string    FOO = ""; }',
        ];

        yield [
            '<?php class Baz {
                public const string FIRST = "1";
                protected const int SECOND = 2;
                private const array THIRD = [];
            }',
            '<?php class Baz {
                public const string  FIRST = "1";
                protected const int   SECOND = 2;
                private const array    THIRD = [];
            }',
        ];

        yield [
            '<?php class Test {
                public function foo(): void {}
                public const string BAR = "bar";
                private $baz = 1;
            }',
            '<?php class Test {
                public function foo(): void {}
                public const string   BAR = "bar";
                private $baz = 1;
            }',
        ];

        yield [
            '<?php class Foo {public const ?string BAR = null; }',
            '<?php class Foo {public const ?string   BAR = null; }',
        ];

        yield [
            '<?php class Foo {public const int|string BAR = 0; }',
            '<?php class Foo {public const int|string    BAR = 0; }',
        ];

        yield [
            '<?php class Foo {public const int /**bla bla*/VALUE = 0; }',
            '<?php class Foo {public const int/**bla bla*/VALUE = 0; }',
        ];

        yield [
            '<?php class Foo {public const int /**bla bla*/ VALUE = 0; }',
            '<?php class Foo {public const int  /**bla bla*/ VALUE = 0; }',
        ];
    }

    /**
     * @dataProvider provideFix84Cases
     *
     * @requires PHP 8.4
     */
    public function testFix84(string $expected, ?string $input = null): void
    {
        $this->fixer->configure(['elements' => ['property', 'constant', 'function']]);
        $this->doTest($expected, $input);
    }

    /**
     * @return iterable<string, array{string, 1?: ?string}>
     */
    public static function provideFix84Cases(): iterable
    {
        yield 'asymmetric visibility' => [
            <<<'PHP'
                <?php
                class Foo
                {
                    public(set) bool $a;
                    protected(set) int $i;
                    private(set) string $s;
                }
                PHP,
            <<<'PHP'
                <?php
                class Foo
                {
                    public(set) bool$a;
                    protected(set) int$i;
                    private(set) string$s;
                }
                PHP,
        ];

        yield 'final property' => [
            '<?php class Foo { final int $i; }',
            '<?php class Foo { final int$i; }',
        ];
    }
}
